/*
 * Decompiled with CFR 0.152.
 */
package com.teamscale.index.findings.abaplint;

import com.teamscale.index.findings.UpdaterUtils;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.Collection;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.jetbrains.annotations.VisibleForTesting;
import org.jsoup.Connection;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;
import org.jsoup.parser.Tag;

public class ABAPLintRuleDescriptionGenerator {
    private static final Logger LOGGER = LogManager.getLogger();
    private static final String ABAP_LINT_RULES_BASE_URL = "https://rules.abaplint.org/";
    private static final String CLEAN_ABAP_STYLE_GUIDE_REPOSITORY_URL = "https://github.com/SAP/styleguides/blob/main/clean-abap/";
    private static final String CLEAN_ABAP_STYLE_GUIDE_URL = "https://github.com/SAP/styleguides/blob/main/clean-abap/CleanABAP.md";
    private static final String CLEAN_ABAP_STYLE_GUIDE_DOWNLOAD_URL = "https://raw.githubusercontent.com/SAP/styleguides/main/clean-abap/CleanABAP.md";
    private static final String CLEAN_ABAP_DESCRIPTION_SECTION_HEADER = "Clean ABAP Code Style Guide";
    private static final Pattern EXAMPLES_SECTION_PATTERN = Pattern.compile("(?s)`(?<examples>\\* Bad example.+)`,");
    private static final Pattern CLEAN_ABAP_SECTION_REFERENCE_AFTER_HEADER_PATTERN = Pattern.compile("> \\[Clean ABAP].*\n");
    private static final Pattern MARKDOWN_LINK_PATTERN = Pattern.compile("\\[(?<header>[^]]*)]\\((?<link>[^)]*)\\)");
    private static final String BAD_EXAMPLE_RAW_HEADER = "* Bad example";
    private static final String GOOD_EXAMPLE_RAW_HEADER = "* Good example";
    private static final Pattern WHITESPACE_CHARACTER_PATTERN = Pattern.compile("\\s");
    @VisibleForTesting
    protected static String cachedCleanAbapStyleGuide = null;
    private final String ruleName;
    private final Path destinationDirectory;

    @VisibleForTesting
    ABAPLintRuleDescriptionGenerator(String ruleName, Path destinationDirectory) {
        Objects.requireNonNull(cachedCleanAbapStyleGuide);
        this.destinationDirectory = destinationDirectory;
        this.ruleName = ruleName;
    }

    static void generateDescriptions(Collection<String> ruleNames, Path ruleDescriptionsDirectoryPath, boolean deleteSuperfluousDescriptions) throws IOException {
        ABAPLintRuleDescriptionGenerator.downloadAndCacheCleanAbapStyleGuide();
        File ruleDescriptionsDirectory = ruleDescriptionsDirectoryPath.toFile();
        if (!ruleDescriptionsDirectory.exists() || !ruleDescriptionsDirectory.isDirectory()) {
            throw new IllegalArgumentException(ruleDescriptionsDirectory.getAbsolutePath() + " is not a valid directory!");
        }
        Set<String> rulesWithNoDescription = ABAPLintRuleDescriptionGenerator.getRulesWithNoDescription(ruleDescriptionsDirectoryPath, ruleNames, deleteSuperfluousDescriptions);
        LOGGER.info("Generating rule descriptions for '{}'", rulesWithNoDescription);
        for (String ruleWithNoDescription : rulesWithNoDescription) {
            new ABAPLintRuleDescriptionGenerator(ruleWithNoDescription, ruleDescriptionsDirectoryPath).generate();
        }
    }

    private static Set<String> getRulesWithNoDescription(Path ruleDescriptionsDirectoryPath, Collection<String> ruleNames, boolean deleteSuperfluousDescriptions) throws IOException {
        HashSet<String> rulesWithNoDescription = new HashSet<String>();
        HashSet<Path> superfluousRules = new HashSet<Path>();
        ABAPLintRuleDescriptionGenerator.categorizeRules(ruleDescriptionsDirectoryPath, ruleNames, rulesWithNoDescription, superfluousRules);
        if (deleteSuperfluousDescriptions) {
            superfluousRules.forEach(ABAPLintRuleDescriptionGenerator::deleteRuleDescription);
        }
        return rulesWithNoDescription;
    }

    private static void categorizeRules(Path ruleDescriptionsDirectoryPath, Collection<String> ruleNames, Set<String> rulesWithNoDescription, Set<Path> superfluousRules) throws IOException {
        rulesWithNoDescription.addAll(ruleNames);
        try (Stream<Path> ruleDescriptionFilePaths = Files.list(ruleDescriptionsDirectoryPath);){
            ruleDescriptionFilePaths.forEach(ruleDescriptionFile -> ABAPLintRuleDescriptionGenerator.categorizeRule(ruleDescriptionFile, rulesWithNoDescription, superfluousRules));
        }
    }

    private static void categorizeRule(Path ruleDescriptionFile, Set<String> rulesWithNoDescription, Set<Path> superfluousRules) {
        boolean isDescriptionSuperfluous;
        String fileName = ruleDescriptionFile.getFileName().toString();
        if (fileName.equals("LICENSE")) {
            return;
        }
        String ruleNameFromFile = fileName.substring(0, fileName.indexOf(46));
        boolean bl = isDescriptionSuperfluous = !rulesWithNoDescription.remove(ruleNameFromFile);
        if (isDescriptionSuperfluous) {
            superfluousRules.add(ruleDescriptionFile);
        }
    }

    private static void deleteRuleDescription(Path ruleDescriptionFile) {
        try {
            Files.delete(ruleDescriptionFile);
        }
        catch (IOException e) {
            LOGGER.error("Unable to delete superfluous rule description", (Throwable)e);
        }
    }

    private void generate() throws IOException {
        Connection connection = Jsoup.connect((String)this.getRuleDescriptionUrl());
        Document description = connection.get();
        String ruleDescriptionContent = this.generateRuleDescriptionFromDocument(description);
        Path filePath = this.destinationDirectory.resolve(this.ruleName + ".md");
        Files.writeString(filePath, (CharSequence)ruleDescriptionContent, StandardOpenOption.CREATE_NEW, StandardOpenOption.WRITE);
    }

    @VisibleForTesting
    String generateRuleDescriptionFromDocument(Document description) {
        Object ruleDescription = "";
        ruleDescription = (String)ruleDescription + this.extractDescriptionSection(description);
        ruleDescription = (String)ruleDescription + this.extractExtendedInformationSection(description);
        ruleDescription = (String)ruleDescription + this.extractExamplesSection(description);
        boolean includesCleanAbapGuideline = false;
        int cleanAbapLinkIndex = ((String)ruleDescription).indexOf("https://github.com/SAP/styleguides/blob/main/clean-abap/CleanABAP.md#");
        if (cleanAbapLinkIndex != -1) {
            String restOfLink = ((String)ruleDescription).substring(cleanAbapLinkIndex + CLEAN_ABAP_STYLE_GUIDE_URL.length() + 1);
            Matcher matcher = WHITESPACE_CHARACTER_PATTERN.matcher(restOfLink);
            String cleanAbapGuidelineHeaderId = !matcher.find() ? restOfLink : restOfLink.substring(0, matcher.start());
            String cleanAbapRuleGuideline = this.extractCleanAbapGuideline(cleanAbapGuidelineHeaderId);
            ruleDescription = (String)ruleDescription + cleanAbapRuleGuideline;
            includesCleanAbapGuideline = !cleanAbapRuleGuideline.isEmpty();
        }
        ruleDescription = (String)ruleDescription + this.createFooter(includesCleanAbapGuideline);
        return ruleDescription;
    }

    private String extractCleanAbapGuideline(String headerId) {
        Matcher referenceInTableOfContentsMatcher = Pattern.compile("\\[(?<header>[^]]*)]\\(#" + headerId + "\\)").matcher(cachedCleanAbapStyleGuide);
        if (!referenceInTableOfContentsMatcher.find()) {
            LOGGER.error("Rule '{}': Could not find reference to section in Clean ABAP style guide", (Object)this.ruleName);
            return "";
        }
        String actualHeader = referenceInTableOfContentsMatcher.group("header");
        Matcher sectionHeaderMatcher = Pattern.compile("\n#+ " + actualHeader).matcher(cachedCleanAbapStyleGuide);
        if (!sectionHeaderMatcher.find()) {
            LOGGER.error("Rule '{}': Could not find section with header '{}' in Clean ABAP style guide", (Object)this.ruleName, (Object)actualHeader);
            return "";
        }
        int ruleStyleGuideStartIndex = sectionHeaderMatcher.end();
        int ruleStyleGuideEndIndex = cachedCleanAbapStyleGuide.indexOf("\n#", ruleStyleGuideStartIndex);
        String ruleStyleGuide = cachedCleanAbapStyleGuide.substring(ruleStyleGuideStartIndex, ruleStyleGuideEndIndex);
        ruleStyleGuide = this.sanitizeCleanAbapRuleStyleGuideMarkdown(ruleStyleGuide);
        return "# %s\n%s\n\n".formatted(CLEAN_ABAP_DESCRIPTION_SECTION_HEADER, ruleStyleGuide);
    }

    private String sanitizeCleanAbapRuleStyleGuideMarkdown(String ruleStyleGuide) {
        return ABAPLintRuleDescriptionGenerator.fixBrokenCleanAbapLinks(this.removeCleanAbapSectionReferenceAfterHeader(ruleStyleGuide));
    }

    private String removeCleanAbapSectionReferenceAfterHeader(String ruleStyleGuide) {
        String result = ruleStyleGuide;
        Matcher sectionReferenceInRuleGuidelineMatcher = CLEAN_ABAP_SECTION_REFERENCE_AFTER_HEADER_PATTERN.matcher(result);
        if (sectionReferenceInRuleGuidelineMatcher.find()) {
            result = result.substring(sectionReferenceInRuleGuidelineMatcher.end()).strip();
        } else {
            LOGGER.warn("Rule '{}': Could not remove Markdown section reference in guideline. Please check the section for Clean ABAP style guide", (Object)this.ruleName);
        }
        return result;
    }

    private static String fixBrokenCleanAbapLinks(String ruleStyleGuide) {
        return MARKDOWN_LINK_PATTERN.matcher(ruleStyleGuide).replaceAll(matchResult -> {
            String header = matchResult.group(1);
            Object link = matchResult.group(2);
            if (((String)link).startsWith("#")) {
                link = CLEAN_ABAP_STYLE_GUIDE_URL + (String)link;
            } else if (!((String)link).startsWith("http")) {
                link = CLEAN_ABAP_STYLE_GUIDE_REPOSITORY_URL + (String)link;
            }
            return "[%s](%s)".formatted(header, link);
        });
    }

    private static void downloadAndCacheCleanAbapStyleGuide() throws IOException {
        cachedCleanAbapStyleGuide = UpdaterUtils.downloadContentFromUrl(CLEAN_ABAP_STYLE_GUIDE_DOWNLOAD_URL, true);
    }

    private String extractDescriptionSection(Document description) {
        return this.extractSimpleHeaderContentSection("Description", description);
    }

    private String extractExtendedInformationSection(Document description) {
        return this.extractSimpleHeaderContentSection("Extended Information", description);
    }

    private String createFooter(boolean includeCleanAbapReference) {
        Object footer = "---\nFor more information visit the [official ABAPLint rule documentation](%s) <br>\n".formatted(this.getRuleDescriptionUrl());
        if (includeCleanAbapReference) {
            footer = (String)footer + "The contents within section `%s` were copied from the [official SAP Code Style Guides repository](https://github.com/SAP/styleguides/).\n".formatted(CLEAN_ABAP_DESCRIPTION_SECTION_HEADER);
        }
        return footer;
    }

    private String extractExamplesSection(Document description) {
        Element examplesHeaderElement = description.body().select("h2:contains(Examples)").first();
        if (examplesHeaderElement == null) {
            return "";
        }
        Element examplesContentElement = examplesHeaderElement.nextElementSiblings().select("script").first();
        if (examplesContentElement == null) {
            LOGGER.error("Rule '{}': Code examples are missing", (Object)this.ruleName);
            return "";
        }
        Matcher exampleMatcher = EXAMPLES_SECTION_PATTERN.matcher(examplesContentElement.html());
        if (!exampleMatcher.find()) {
            LOGGER.error("Rule '{}': Code examples do not match expected format", (Object)this.ruleName);
            return "";
        }
        return ABAPLintRuleDescriptionGenerator.extractExamplesSection(exampleMatcher.group("examples"));
    }

    @VisibleForTesting
    static String extractExamplesSection(String examples) {
        String badExample;
        int indexOfBadExampleHeader = examples.indexOf(BAD_EXAMPLE_RAW_HEADER);
        int indexOfGoodExampleHeader = examples.indexOf(GOOD_EXAMPLE_RAW_HEADER);
        String goodExample = null;
        if (indexOfBadExampleHeader == -1 && indexOfGoodExampleHeader == -1) {
            throw new IllegalArgumentException("Rule '{}': Examples must at least include bad example");
        }
        if (indexOfGoodExampleHeader == -1) {
            badExample = examples.substring(indexOfBadExampleHeader + BAD_EXAMPLE_RAW_HEADER.length());
        } else {
            badExample = examples.substring(indexOfBadExampleHeader + BAD_EXAMPLE_RAW_HEADER.length(), indexOfGoodExampleHeader);
            goodExample = examples.substring(indexOfGoodExampleHeader + GOOD_EXAMPLE_RAW_HEADER.length());
        }
        return "# Examples\n%s\n%s\n".formatted(ABAPLintRuleDescriptionGenerator.formatCodeExample("Bad example", badExample), ABAPLintRuleDescriptionGenerator.formatCodeExample("Good example", goodExample));
    }

    private static String formatCodeExample(String header, @Nullable String codeExample) {
        if (codeExample == null) {
            return "";
        }
        return "**%s**\n```abap\n%s\n```\n".formatted(header, codeExample.strip());
    }

    private String extractSimpleHeaderContentSection(String sectionHeader, Document description) {
        String sectionContent;
        Element sectionHeaderElement = description.body().select("h2:contains(" + sectionHeader + ")").first();
        if (sectionHeaderElement == null) {
            return "";
        }
        Element sectionContentElement = sectionHeaderElement.nextElementSibling();
        if (sectionContentElement == null || !sectionContentElement.tag().equals((Object)Tag.valueOf((String)"p"))) {
            LOGGER.warn("Rule '{}': {} is missing", (Object)this.ruleName, (Object)sectionHeader);
            sectionContent = "";
        } else {
            sectionContent = sectionContentElement.wholeText().strip();
        }
        return "# %s\n%s\n\n".formatted(sectionHeader, sectionContent);
    }

    private String getRuleDescriptionUrl() {
        return ABAP_LINT_RULES_BASE_URL + this.ruleName;
    }
}

