<h2>Why is this an issue?</h2>
<p>In JavaScript, a promise is an object representing the eventual completion or failure of an asynchronous operation. It is a way to handle
asynchronous operations more elegantly and avoid the "callback hell".</p>
<p>A promise can be in one of three states:</p>
<ul>
  <li> Pending: The initial state of a promise. It represents that the asynchronous operation is still ongoing and has not yet been fulfilled or
  rejected. </li>
  <li> Fulfilled: The state of a promise when the asynchronous operation has been successfully completed. It represents that the promised value is
  available and can be consumed. </li>
  <li> Rejected: The state of a promise when the asynchronous operation encounters an error or fails to complete. It represents that an error has
  occurred and the promised value is not available. </li>
</ul>
<p>The basic syntax for creating a promise in JavaScript is as follows:</p>

```js
const myPromise = new Promise((resolve, reject) => {
  // Asynchronous operation
  // If the operation is successful, call resolve(value)
  // If the operation fails, call reject(error)
});
```
<p>However, when it comes to immediately resolving or rejecting states, creating a new promise with the <code>Promise</code> constructor and manually
calling <code>resolve</code> or <code>reject</code> makes the code verbose and more difficult to read.</p>

```js
const result = new Promise(resolve => resolve(42)); // Noncompliant: Redundant to explicitly create a promise to resolve 42
result.then(value => {
  console.log(value); // Output: 42
});
```
<p>Instead, a promise can be created with <code>Promise.resolve</code>. It is typically used when you want to create a new promise that is already
resolved with a certain value. It is commonly used to wrap synchronous values or functions into promises.</p>
<h2>How to fix it</h2>
<p>If you already have a synchronous value that you want to convert into a promise, using <code>Promise.resolve</code> is more concise and
straightforward. It immediately creates a promise that is already resolved with the provided value.</p>

```js
const result = Promise.resolve(42);
result.then(value => {
  console.log(value); // Output: 42
});
```
<p>Similarly, if you have an error or an exceptional condition and want to create a promise that is immediately rejected with that error, using
<code>Promise.reject</code> is more straightforward. It creates a promise in the rejected state with the provided error.</p>

```js
const error = new Error('Something went wrong');
const promise = Promise.reject(error);
```
<p>If you have a condition and want to create a promise that is either resolved or rejected based on that condition, using
<code>Promise.resolve</code> or <code>Promise.reject</code> helps make the code more readable and concise.</p>

```js
function fetchData() {
  if (cache) {
    return Promise.resolve(cache);
  } else if (shouldFetchData()) {
    return fetchDataFromServer()
        .then(data => {
            cache = data;
            return data;
        });
  } else {
    return Promise.reject(new Error('Data fetch is not required'));
  }
}
```
<p>When you have a promise chain and want to introduce an intermediate step with an immediately resolved value, using <code>Promise.resolve</code>
allows you to continue the chain without introducing unnecessary complexity.</p>

```js
const data = cache ? cache : fetchData();

return Promise.resolve(data) // data may be a Promise or not, we need to wrap it
    .then(data => {
        return sanitizeData(data);
    })
```
<p>Using <code>Promise.resolve</code> and <code>Promise.reject</code> is particularly useful when you want to simplify the creation of promises with
immediately resolved or rejected states. They provide a cleaner and more direct approach compared to creating a new promise with the
<code>Promise</code> constructor and manually calling <code>resolve</code> or <code>reject</code>.</p>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Learn/JavaScript/Asynchronous">Asynchronous JavaScript</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Promise">Promise</a> </li>
  <li> MDN web docs - <a
  href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Promise/then">Promise.prototype.then()</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Promise/resolve">Promise.resolve()</a>
  </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Promise/reject">Promise.reject()</a>
  </li>
</ul>