<h2>Why is this an issue?</h2>
<p>Using <code>Function.prototype.bind</code> and arrows functions as attributes will negatively impact performance in React. Each time the parent is
rendered, the function will be re-created and trigger a render of the component causing excessive renders and more memory use. Wrapping the function
in a <code>useCallback</code> hook will avoid additional renders. This rule ignores <code>Refs</code>. This rule does not raise findings on DOM nodes
since that may require wrapping the DOM in a component. Still, better performance can be achieved if this rule is respected in DOM nodes too.</p>
<h3>Noncompliant code example</h3>

```js
<Component onClick={this._handleClick.bind(this)}></Component>

<Component onClick={() => handleClick()}></Component>
```
<h3>Compliant solution</h3>

```js
function handleClick() {
    //...
}

<Component onClick={handleClick}></Component>
```
<p>Situation can become more complicated when you need to pass additional parameters to the handler. Consider following component printing the list of
letters. Consider following non-compliant example</p>

```js
class Alphabet extends React.Component {
    handleClick(letter) {
        console.log(`clicked ${letter}`);
    }
    render() {
        return (<div><ul>
            {letters.map(letter =>
                <li key={letter} onClick={() => this.handleClick(letter)}>{letter}</li>
            )}
        </ul></div>)
    }
}
```
<p>To avoid creating the arrow function you can factor out <code>li</code> element as separate child component and use <code>props</code> to pass the
<code>letter</code> and <code>onClick</code> handler.</p>

```js
class Alphabet extends React.Component {
    handleClick(letter) {
        console.log(`clicked ${letter}`);
    }
    render() {
        return (<div><ul>
            {letters.map(letter =>
                <Letter key={letter} letter={letter} handleClick={this.handleClick}></Letter>
            )}
        </ul></div>)
    }
}

class Letter extends React.Component {
    constructor(props) {
        super(props);
        this.handleClick = this.handleClick.bind(this)
    }
    handleClick() {
        this.props.handleClick(this.props.letter);
    }
    render() {
        return <li onClick={this.handleClick}> {this.props.letter} </li>
    }
}
```
<p>alternatively you could rewrite <code>Letter</code> as a function and use <code>useCallback</code></p>

```js
function Letter({ handleClick, letter }) {
    const onClick = React.useCallback(() => handleClick(letter), [letter])

    return <li onClick={onClick}>{letter}</li>
}
```
<h2>Resources</h2>
<ul>
  <li> <a href="https://reactjs.org/docs/faq-functions.html">Passing Functions to Components</a> - React documentation </li>
</ul>