<p>The improper storage of passwords poses a significant security risk to software applications. This vulnerability arises when passwords are stored
in plaintext or with a fast hashing algorithm. To exploit this vulnerability, an attacker typically requires access to the stored passwords.</p>
<h2>Why is this an issue?</h2>
<p>Attackers who would get access to the stored passwords could reuse them without further attacks or with little additional effort.<br> Obtaining the
plaintext passwords, they could then gain unauthorized access to user accounts, potentially leading to various malicious activities.</p>
<h3>What is the potential impact?</h3>
<p>Plaintext or weakly hashed password storage poses a significant security risk to software applications.</p>
<h4>Unauthorized Access</h4>
<p>When passwords are stored in plaintext or with weak hashing algorithms, an attacker who gains access to the password database can easily retrieve
and use the passwords to gain unauthorized access to user accounts. This can lead to various malicious activities, such as unauthorized data access,
identity theft, or even financial fraud.</p>
<h4>Credential Reuse</h4>
<p>Many users tend to reuse passwords across multiple platforms. If an attacker obtains plaintext or weakly hashed passwords, they can potentially use
these credentials to gain unauthorized access to other accounts held by the same user. This can have far-reaching consequences, as sensitive personal
information or critical systems may be compromised.</p>
<h4>Regulatory Compliance</h4>
<p>Many industries and jurisdictions have specific regulations and standards to protect user data and ensure its confidentiality. Storing passwords in
plaintext or with weak hashing algorithms can lead to non-compliance with these regulations, potentially resulting in legal consequences, financial
penalties, and damage to the reputation of the software application and its developers.</p>
<h2>How to fix it in Spring</h2>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>
<p>The following code is vulnerable because it uses a legacy digest-based password encoding that is not considered secure.</p>

```java
@Autowired
public void configureGlobal(AuthenticationManagerBuilder auth, DataSource dataSource) throws Exception {
  auth.jdbcAuthentication()
    .dataSource(dataSource)
    .usersByUsernameQuery("SELECT * FROM users WHERE username = ?")
    .passwordEncoder(new StandardPasswordEncoder()); // Noncompliant
}
```
<h4>Compliant solution</h4>

```java
@Autowired
public void configureGlobal(AuthenticationManagerBuilder auth, DataSource dataSource) throws Exception {
  auth.jdbcAuthentication()
    .dataSource(dataSource)
    .usersByUsernameQuery("SELECT * FROM users WHERE username = ?")
    .passwordEncoder(new BCryptPasswordEncoder());
}
```
<h3>How does this work?</h3>
<h4>Use secure password hashing algorithms</h4>
<p>In general, you should rely on an algorithm that has no known security vulnerabilities. The MD5 and SHA-1 algorithms should not be used.</p>
<p>Some algorithms, such as the SHA family functions, are considered strong for some use cases, but are too fast in computation and therefore
vulnerable to brute force attacks, especially with bruteforce-attack-oriented hardware.</p>
<p>To protect passwords, it is therefore important to choose modern, slow password-hashing algorithms. The following algorithms are, in order of
strength, the most secure password hashing algorithms to date:</p>
<ol>
  <li> Argon2 </li>
  <li> scrypt </li>
  <li> bcrypt </li>
  <li> PBKDF2 </li>
</ol>
<p>Argon2 should be the best choice, and others should be used when the previous one is not available. For systems that must use FIPS-140-certified
algorithms, PBKDF2 should be used.</p>
<p>Whenever possible, choose the strongest algorithm available. If the algorithm currently used by your system should be upgraded, OWASP documents
possible upgrade methods here: <a
href="https://cheatsheetseries.owasp.org/cheatsheets/Password_Storage_Cheat_Sheet.html#upgrading-legacy-hashes">Upgrading Legacy Hashes</a>.</p>
<p>In the previous example, the <code>BCryptPasswordEncoder</code> is a password hashing function in Java that is designed to be secure and resistant
to various types of attacks, including brute-force and rainbow table attacks. It is slow, adaptative, and automatically implements a salt.</p>
<h4>Never store passwords in plaintext</h4>
<p>A user password should never be stored in plaintext. Instead, a hash should be produced from it using a secure algorithm. When dealing with
password storage security, best practices recommend relying on a slow hashing algorithm, that will make brute force attacks more difficult. Using a
hashing function with adaptable computation and memory complexity also is recommended to be able to increase the security level with time.</p>
<p>Adding a salt to the digest computation is also recommended to prevent pre-computed table attacks (see rule <a href='checks/checks/?languages=JAVA&tools=SONAR_LINT&rule=java:S2053'>java:S2053</a>).</p>
<h3>Pitfalls</h3>
<h4>Pre-hashing passwords</h4>
<p>As bcrypt has a maximum length input length of 72 bytes for most implementations, some developers may be tempted to pre-hash the password with a
stronger algorithm before hashing it with bcrypt.</p>
<p>Pre-hashing passwords with bcrypt is not recommended as it can lead to a specific range of issues. Using a strong salt and a high number of rounds
is enough to protect the password.</p>
<p>More information about this can be found here: <a
href="https://cheatsheetseries.owasp.org/cheatsheets/Password_Storage_Cheat_Sheet.html#pre-hashing-passwords-with-bcrypt">Pre-hashing Passwords with
Bcrypt</a>.</p>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> Spring Framework Security Documentation - <a
  href="https://docs.spring.io/spring-security/site/docs/current/api/org/springframework/security/crypto/bcrypt/BCryptPasswordEncoder.html">Class
  BCryptPasswordEncoder</a> </li>
  <li> OWASP CheatSheet - <a href="https://cheatsheetseries.owasp.org/cheatsheets/Password_Storage_Cheat_Sheet.html">Password Storage Cheat Sheet</a>
  </li>
</ul>
<h3>Standards</h3>
<ul>
  <li> OWASP - <a href="https://owasp.org/Top10/A02_2021-Cryptographic_Failures/">Top 10 2021 Category A2 - Cryptographic Failures</a> </li>
  <li> OWASP - <a href="https://owasp.org/Top10/A04_2021-Insecure_Design/">Top 10 2021 Category A4 - Insecure Design</a> </li>
  <li> OWASP - <a href="https://owasp.org/www-project-top-ten/2017/A3_2017-Sensitive_Data_Exposure">Top 10 2017 Category A3 - Sensitive Data
  Exposure</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/256">CWE-256 - Plaintext Storage of a Password</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/916">CWE-916 - Use of Password Hash With Insufficient Computational Effort</a> </li>
  <li> STIG Viewer - <a href="https://stigviewer.com/stig/application_security_and_development/2023-06-08/finding/V-222542">Application Security and
  Development: V-222542</a> - The application must only store cryptographic representations of passwords. </li>
</ul>