/*
 * Decompiled with CFR 0.152.
 */
package org.conqat.lib.commons.io;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.conqat.lib.commons.error.FormatException;
import org.conqat.lib.commons.io.ProcessUtils;
import org.conqat.lib.commons.string.StringUtils;
import org.conqat.lib.commons.system.SystemUtils;
import org.conqat.lib.commons.version.Version;

public class DotNetUtils {
    private static final Logger LOGGER = LogManager.getLogger(DotNetUtils.class);
    public static final String MONO_COMMAND = "mono";
    private static final String DOTNET_COMMAND = "dotnet";
    private static final Pattern DOTNET_RUNTIME_VERSION_PATTERN = Pattern.compile("Microsoft\\.NETCore\\.App\\s+?([0-9.]+)\\s+?\\[.+\\]");
    private static final Version DOTNET_MIN_VERSION = new Version(6, 0);

    public static ProcessBuilder createDotNetProcessBuilder(String ... arguments) {
        ProcessBuilder builder = new ProcessBuilder(new String[0]);
        if (!SystemUtils.isWindows()) {
            builder.command().add(MONO_COMMAND);
        }
        builder.command().addAll(Arrays.asList(arguments));
        return builder;
    }

    public static @Nullable String checkDotnetIsInstalled() {
        try {
            ProcessUtils.ExecutionResult result = ProcessUtils.execute(new String[]{DOTNET_COMMAND, "--list-runtimes"});
            String errorMessage = DotNetUtils.performDotnetCheck(result.getReturnCode(), result.getStdout(), result.getStderr());
            if (errorMessage != null) {
                DotNetUtils.logDotNetInstallationDetails(result);
            }
            return errorMessage;
        }
        catch (IOException e) {
            DotNetUtils.logDotNetInstallationDetails(null);
            return String.format("Error during the detection of the .NET runtime. `dotnet` could not be executed. The .NET runtime version %s or higher has to be installed.\nFurther details: %s", DOTNET_MIN_VERSION, e.getMessage());
        }
    }

    private static void logDotNetInstallationDetails(ProcessUtils.ExecutionResult executionResult) {
        LOGGER.error(".NET installation could not be detected or did not fulfill requirements. This log will contain additional warning messages to simplify debugging.");
        String path = System.getenv("PATH");
        LOGGER.warn("Value of $PATH: " + path);
        Optional<File> fullPath = DotNetUtils.findUsedDotNetInstallation(path);
        if (!fullPath.isPresent()) {
            LOGGER.warn("Could not find dotnet executable on the path!");
        } else {
            LOGGER.warn("First installation found on PATH: " + fullPath.get().getAbsolutePath());
            LOGGER.warn("Can execute: " + fullPath.get().canExecute());
        }
        if (executionResult != null) {
            LOGGER.warn("Execution returned error code " + executionResult.getReturnCode());
            LOGGER.warn("Std out: " + executionResult.getStdout());
            LOGGER.warn("Std err: " + executionResult.getStderr());
        }
    }

    private static Optional<File> findUsedDotNetInstallation(String path) {
        for (String directory : path.split(File.pathSeparator)) {
            File file = new File(directory, DOTNET_COMMAND);
            if (file.isFile()) {
                return Optional.of(file);
            }
            file = new File(directory, "dotnet.exe");
            if (!file.isFile()) continue;
            return Optional.of(file);
        }
        return Optional.empty();
    }

    static @Nullable String performDotnetCheck(int returnCode, @NonNull String stdout, @NonNull String stderr) {
        if (returnCode != 0) {
            return String.format("Error executing `dotnet` (%d). %s", returnCode, DotNetUtils.generateGenericInstallErrorMessage(stdout, stderr));
        }
        List<Version> installedVersions = DotNetUtils.extractRuntimeVersions(stdout);
        Optional highestVersion = installedVersions.stream().max(Comparator.naturalOrder());
        if (!highestVersion.isPresent()) {
            return String.format("No .NET runtime detected. %s", DotNetUtils.generateGenericInstallErrorMessage(stdout, stderr));
        }
        if (!((Version)highestVersion.get()).isGreaterOrEqual(DOTNET_MIN_VERSION)) {
            return String.format("Unsupported .NET runtime detected (latest: %s). %s", highestVersion.get(), DotNetUtils.generateGenericInstallErrorMessage(stdout, stderr));
        }
        return null;
    }

    private static @NonNull String generateGenericInstallErrorMessage(String stdout, String stderr) {
        return String.format("Please make sure that version %s or higher of the .NET runtime is installed.\nstd out: %s\nstd error: %s", DOTNET_MIN_VERSION, StringUtils.isEmpty(stdout) ? "-empty-" : stdout, StringUtils.isEmpty(stderr) ? "-empty-" : stderr);
    }

    public static ProcessUtils.ExecutionResult executeDotNet(String[] completeArguments) throws IOException {
        return ProcessUtils.execute(DotNetUtils.createDotNetProcessBuilder(completeArguments));
    }

    static @NonNull List<Version> extractRuntimeVersions(@NonNull String rawVersionsList) {
        return StringUtils.splitLinesAsList(rawVersionsList).stream().map(DotNetUtils::extractRuntimeVersionFromLine).filter(Objects::nonNull).collect(Collectors.toList());
    }

    static @Nullable Version extractRuntimeVersionFromLine(@NonNull String line) {
        Matcher matcher = DOTNET_RUNTIME_VERSION_PATTERN.matcher(line);
        if (!matcher.find() || matcher.groupCount() != 1) {
            return null;
        }
        try {
            return Version.parseVersion(matcher.group(1));
        }
        catch (FormatException e) {
            return null;
        }
    }
}

