Certain exceptions should not be caught, because it is considered bad practise.
Catching such exceptions should be avoided.

# What Does This Check Look For?
This check identifies catch clauses that catch certain exceptions.
The list of exceptions to avoid catching can be customized.
By default, this list includes classes such as `IndexOutOfRangeException`, `NullReferenceException`, or `DivideByZeroException`.

# Why is This a Problem?
Catching exceptions like `IndexOutOfRangeException`, `NullReferenceException`, or `DivideByZeroException` is unusual in normal code.
These exceptions usually signal programming errors or logical flaws that should be addressed directly in the code rather than being circumvented through exception handling.
Ignoring the root cause of such exceptions and relying on catch blocks for control flow can lead to code that is less reliable, more difficult to debug, and potentially more vulnerable to errors.

# How Can I Resolve This?
The Problem can be resolved by fixing the underlying issue or using checks instead of exceptions.

# Examples
## Non-Compliant
A `NullReferenceException` will be thrown by `list[0]`, if `GetFirst()` is called before `SetList()`.
```cs
class Foo
{
	private List<string> list;

	public string GetFirst()
	{
		try
		{
			return list[0];
		}
		catch (NullReferenceException e)
		{
			// handle exception
		}
	}

	public void SetList(List<string> list)
	{
		this.list = list;
	}
}
```

## Compliant
With inserting a null check before `list[0]`.
```cs
class Foo
{
	private List<string> list;

	public string GetFirst()
	{
		if (list == null)
		{
			// handle null value
		}
		return list[0];
	}

	public void SetList(List<string> list)
	{
		this.list = list;
	}
}
```
Even better, directly fix the underlying problem.
In this example by initializing the list in the constructor.

```cs
class Foo
{
	private readonly List<string> list;

	public Foo(List<string> list)
	{
		this.list = list;
	}

	public string GetFirst()
	{
		return list[0];
	}

	public void AddToList(List<string> list)
	{
		this.list.AddRange(list);
	}
}
```

# Where Can I Learn More?
- [Refactoring Guru: Replace Exception with Test](https://refactoring.guru/replace-exception-with-test)