/*
 * Decompiled with CFR 0.152.
 */
package org.conqat.lib.commons.filesystem;

import com.google.common.base.Preconditions;
import java.io.BufferedInputStream;
import java.io.Closeable;
import java.io.EOFException;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.CopyOption;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Properties;
import java.util.concurrent.atomic.LongAdder;
import java.util.function.Consumer;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.GZIPInputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.conqat.lib.commons.assertion.CCSMAssert;
import org.conqat.lib.commons.collections.CollectionUtils;
import org.conqat.lib.commons.collections.PairList;
import org.conqat.lib.commons.concurrent.ThreadUtils;
import org.conqat.lib.commons.date.DateTimeUtils;
import org.conqat.lib.commons.filesystem.EByteOrderMark;
import org.conqat.lib.commons.filesystem.FilenameComparator;
import org.conqat.lib.commons.filesystem.NonThrowingFileVisitor;
import org.conqat.lib.commons.filesystem.PathBasedContentProviderBase;
import org.conqat.lib.commons.filesystem.TemporaryDirectory;
import org.conqat.lib.commons.filesystem.ZipFile;
import org.conqat.lib.commons.function.SupplierWithException;
import org.conqat.lib.commons.regex.Patterns;
import org.conqat.lib.commons.string.StringUtils;

public final class FileSystemUtils {
    public static final Charset SYSTEM_CHARSET = Charset.forName(System.getProperty("native.encoding", Charset.defaultCharset().name()));
    public static final String TEMP_DIR_PATH = System.getProperty("java.io.tmpdir");
    public static final char UNIX_SEPARATOR = '/';
    private static final char WINDOWS_SEPARATOR = '\\';
    private static final String METRIC_SYSTEM_PREFIXES = "KMGTPEZ";
    private static final Pattern DATA_SIZE_UNIT_START_PATTERN = Pattern.compile("[^\\d\\s.,]");
    private static final Pattern FILENAME_PLACEHOLDERS = Pattern.compile("[-_]+");

    public static int copy(InputStream input, OutputStream output) throws IOException {
        int len;
        byte[] buffer = new byte[8192];
        int size = 0;
        while ((len = input.read(buffer)) > 0) {
            output.write(buffer, 0, len);
            size += len;
        }
        return size;
    }

    @Deprecated
    public static void copyFile(File sourceFile, File targetFile) throws IOException {
        FileSystemUtils.copyFile(sourceFile.toPath(), targetFile.toPath());
    }

    public static void copyFile(Path sourceFile, Path targetFile) throws IOException {
        if (sourceFile.toAbsolutePath().equals(targetFile.toAbsolutePath())) {
            throw new IOException("Can not copy file onto itself: " + String.valueOf(sourceFile));
        }
        FileSystemUtils.ensureParentDirectoryExists(targetFile);
        Files.copy(sourceFile, targetFile, new CopyOption[0]);
    }

    @Deprecated
    public static int copyFiles(File sourceDirectory, File targetDirectory, FileFilter fileFilter) throws IOException {
        List<File> files = FileSystemUtils.listFilesRecursively(sourceDirectory, fileFilter);
        int fileCount = 0;
        for (File sourceFile : files) {
            if (!sourceFile.isFile()) continue;
            String path = sourceFile.getAbsolutePath();
            int index = sourceDirectory.getAbsolutePath().length();
            String newPath = path.substring(index);
            File targetFile = new File(targetDirectory, newPath);
            FileSystemUtils.copyFile(sourceFile, targetFile);
            ++fileCount;
        }
        return fileCount;
    }

    @Deprecated
    public static void deleteRecursively(File directory) throws IOException {
        Preconditions.checkArgument((directory != null ? 1 : 0) != 0, (Object)String.format("Parameter %s is null!", "directory"));
        FileSystemUtils.deleteRecursively(directory.toPath());
    }

    public static void deleteRecursively(@Nullable Path directory) throws IOException {
        if (directory == null) {
            return;
        }
        if (!Files.exists(directory, new LinkOption[0])) {
            return;
        }
        try (Stream<Path> walk = Files.walk(directory, new FileVisitOption[0]);){
            Iterator it = walk.sorted(Collections.reverseOrder()).iterator();
            while (it.hasNext()) {
                Files.delete((Path)it.next());
            }
        }
    }

    @Deprecated
    public static void deleteFile(File file) throws IOException {
        if (!file.exists()) {
            return;
        }
        Files.delete(file.toPath());
    }

    @Deprecated
    public static void renameFileTo(File file, File dest) throws IOException {
        Files.move(file.toPath(), dest.toPath(), new CopyOption[0]);
    }

    @Deprecated
    public static void mkdirs(File dir) throws IOException {
        if (dir.exists() && dir.isDirectory()) {
            return;
        }
        if (!dir.mkdirs()) {
            throw new IOException("Could not create directory " + String.valueOf(dir));
        }
    }

    @Deprecated
    public static void ensureDirectoryExists(File directory) throws IOException {
        FileSystemUtils.ensureDirectoryExists(directory.toPath());
    }

    public static void ensureDirectoryExists(Path directory) throws IOException {
        if (!Files.exists(directory, new LinkOption[0])) {
            Files.createDirectories(directory, new FileAttribute[0]);
        }
        if (Files.exists(directory, new LinkOption[0]) && Files.isWritable(directory)) {
            return;
        }
        Instant start = DateTimeUtils.now();
        while (!(Files.exists(directory, new LinkOption[0]) && Files.isWritable(directory) || start.until(DateTimeUtils.now(), ChronoUnit.MILLIS) >= 100L)) {
            ThreadUtils.sleep(10L);
        }
        if (!Files.exists(directory, new LinkOption[0])) {
            throw new IOException("Directory '" + String.valueOf(directory) + "' could not be created.");
        }
        if (!Files.isWritable(directory)) {
            throw new IOException("Directory '" + String.valueOf(directory) + "' exists, but is not writable.");
        }
    }

    @Deprecated
    public static void ensureParentDirectoryExists(File file) throws IOException {
        FileSystemUtils.ensureParentDirectoryExists(file.toPath());
    }

    public static void ensureParentDirectoryExists(Path file) throws IOException {
        Preconditions.checkArgument((file != null ? 1 : 0) != 0, (Object)String.format("Parameter %s must not be null.", "file"));
        FileSystemUtils.ensureDirectoryExists(file.toAbsolutePath().getParent());
    }

    @Deprecated
    public static List<File> listFilesRecursively(File directory) {
        return FileSystemUtils.listFilesRecursively(directory, null);
    }

    @Deprecated
    public static List<File> listFilesRecursively(File directory, FileFilter filter) {
        if (directory == null || !directory.isDirectory()) {
            return CollectionUtils.emptyList();
        }
        ArrayList<File> result = new ArrayList<File>();
        FileSystemUtils.listFilesRecursively(directory, result, filter);
        return result;
    }

    private static void listFilesRecursively(File directory, Collection<File> result, FileFilter filter) {
        File[] files = directory.listFiles();
        if (files == null) {
            return;
        }
        for (File file : files) {
            if (file.isDirectory()) {
                FileSystemUtils.listFilesRecursively(file, result, filter);
            }
            if (filter != null && !filter.accept(file)) continue;
            result.add(file);
        }
    }

    public static List<String> listFilesInSameLocationForURL(URL baseUrl) throws IOException {
        return FileSystemUtils.listFilesInSameLocationForURL(baseUrl, false);
    }

    public static List<String> listFilesInSameLocationForURL(URL baseUrl, boolean includeSubfolders) throws IOException {
        String protocol = baseUrl.getProtocol();
        if ("file".equals(protocol)) {
            return FileSystemUtils.listFilesForFileURL(baseUrl, includeSubfolders);
        }
        if ("jar".equals(protocol)) {
            return FileSystemUtils.listFilesForJarURL(baseUrl, includeSubfolders);
        }
        throw new IOException("Unsupported protocol: " + protocol);
    }

    private static String getJarUrlParentDirectoryPrefix(URL baseUrl) {
        String parentPath = StringUtils.getLastPart(baseUrl.getPath(), '!');
        parentPath = (parentPath = StringUtils.stripPrefix(parentPath, "/")).endsWith(".class") ? StringUtils.stripSuffix(parentPath, StringUtils.getLastPart(parentPath, '/')) : StringUtils.ensureEndsWith(parentPath, String.valueOf('/'));
        return parentPath;
    }

    private static List<String> listFilesForJarURL(URL baseUrl, boolean recursive) throws IOException {
        try (JarFile jarFile = new JarFile(FileSystemUtils.extractJarFileFromJarURL(baseUrl));){
            String parentPath = FileSystemUtils.getJarUrlParentDirectoryPrefix(baseUrl);
            List<String> list = jarFile.stream().filter(entry -> FileSystemUtils.shouldBeContainedInResult(entry, parentPath, recursive)).map(entry -> StringUtils.stripPrefix(entry.getName(), parentPath)).collect(Collectors.toList());
            return list;
        }
    }

    private static boolean shouldBeContainedInResult(JarEntry entry, String path, boolean recursive) {
        if (entry.isDirectory()) {
            return false;
        }
        String simpleName = StringUtils.getLastPart(entry.getName(), '/');
        String entryPath = StringUtils.stripSuffix(entry.getName(), simpleName);
        return !recursive && entryPath.equals(path) || recursive && entryPath.startsWith(path);
    }

    private static List<String> listFilesForFileURL(URL baseUrl, boolean includeSubfolders) throws IOException {
        try {
            File directory = new File(baseUrl.toURI());
            if (!directory.isDirectory()) {
                directory = directory.getParentFile();
            }
            if (directory == null || !directory.isDirectory()) {
                throw new IOException("Parent directory does not exist or is not readable for " + String.valueOf(baseUrl));
            }
            if (includeSubfolders) {
                File finalDirectory = directory;
                return CollectionUtils.filterAndMap(FileSystemUtils.listFilesRecursively(directory), File::isFile, file -> {
                    String relativeFilePath = StringUtils.stripPrefix(file.getAbsolutePath(), finalDirectory.getAbsolutePath());
                    relativeFilePath = FileSystemUtils.normalizeSeparators(relativeFilePath);
                    return StringUtils.stripPrefix(relativeFilePath, String.valueOf('/'));
                });
            }
            File[] files = directory.listFiles();
            if (files == null) {
                throw new IOException("Failed to list files for directory '" + String.valueOf(directory.toPath()) + "', even though it was supposed to be a valid directory.");
            }
            ArrayList<String> names = new ArrayList<String>();
            for (File file2 : files) {
                if (!file2.isFile()) continue;
                names.add(file2.getName());
            }
            return names;
        }
        catch (URISyntaxException e) {
            throw new IOException("Could not convert URL to valid file: " + String.valueOf(baseUrl), e);
        }
    }

    public static List<String> listTopLevelClassesInJarFile(File jarFile) throws IOException {
        ArrayList<String> result = new ArrayList<String>();
        try (PathBasedContentProviderBase provider = PathBasedContentProviderBase.createProvider(jarFile);){
            Collection<String> paths = provider.getPaths();
            for (String path : paths) {
                if (!path.endsWith(".class") || path.contains("$")) continue;
                String fqn = StringUtils.removeLastPart(path, '.');
                fqn = fqn.replace('/', '.');
                result.add(fqn);
            }
            ArrayList<String> arrayList = result;
            return arrayList;
        }
    }

    public static @Nullable String getFileExtension(Path file) {
        return FileSystemUtils.getFileExtension(file.toString());
    }

    @Deprecated
    public static @Nullable String getFileExtension(File file) {
        return FileSystemUtils.getFileExtension(file.getName());
    }

    public static String getFileExtension(String path) {
        int posLastDot = path.lastIndexOf(46);
        if (posLastDot < 0) {
            return null;
        }
        return path.substring(posLastDot + 1);
    }

    @Deprecated
    public static String getFilenameWithoutExtension(File file) {
        return FileSystemUtils.getFilenameWithoutExtension(file.getName());
    }

    public static String getFilenameWithoutExtension(Path file) {
        return FileSystemUtils.getFilenameWithoutExtension(file.getFileName().toString());
    }

    public static String getFilenameWithoutExtension(String filename) {
        return StringUtils.removeLastPart(filename, '.');
    }

    public static String getLastPathSegment(String filePath) {
        String[] split = FileSystemUtils.getPathSegments(filePath);
        return split[split.length - 1];
    }

    public static String[] getPathSegments(String filePath) {
        return FileSystemUtils.normalizeSeparators(filePath).split(String.valueOf('/'));
    }

    public static boolean isValidPath(String path) {
        try {
            Paths.get(path, new String[0]);
        }
        catch (InvalidPathException ex) {
            return false;
        }
        return Arrays.stream(path.split(Pattern.quote(File.separator))).noneMatch(pathSegment -> pathSegment.contains(String.valueOf('\\')) || pathSegment.contains(String.valueOf('/')));
    }

    public static boolean isPathWriteable(Path file) {
        Path folderToCheck = file;
        if (!Files.isDirectory(file, new LinkOption[0])) {
            folderToCheck = file.toAbsolutePath().getParent();
        }
        while (folderToCheck != null && !Files.isDirectory(folderToCheck, new LinkOption[0])) {
            folderToCheck = folderToCheck.getParent();
        }
        return folderToCheck != null && Files.isWritable(folderToCheck);
    }

    public static String readFileUTF8(Path file) throws IOException {
        return FileSystemUtils.readFile(file, StandardCharsets.UTF_8);
    }

    @Deprecated
    public static String readFileUTF8(File file) throws IOException {
        return FileSystemUtils.readFile(file, StandardCharsets.UTF_8);
    }

    @Deprecated
    public static String readFile(File file, Charset encoding) throws IOException {
        byte[] buffer = FileSystemUtils.readFileBinary(file);
        return StringUtils.bytesToString(buffer, encoding);
    }

    public static String readFile(Path file, Charset encoding) throws IOException {
        byte[] buffer = Files.readAllBytes(file);
        return StringUtils.bytesToString(buffer, encoding);
    }

    public static List<String> readLines(Path file, Charset encoding) throws IOException {
        return StringUtils.splitLinesAsList(FileSystemUtils.readFile(file, encoding));
    }

    public static List<String> readLinesUTF8(Path file) throws IOException {
        return FileSystemUtils.readLines(file, StandardCharsets.UTF_8);
    }

    @Deprecated
    public static List<String> readLinesUTF8(File file) throws IOException {
        return FileSystemUtils.readLinesUTF8(file.toPath());
    }

    public static byte[] readFileBinary(String filePath) throws IOException {
        return Files.readAllBytes(Paths.get(filePath, new String[0]));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Deprecated
    public static byte[] readFileBinary(File file) throws IOException {
        FileInputStream in = new FileInputStream(file);
        byte[] buffer = new byte[(int)file.length()];
        ByteBuffer byteBuffer = ByteBuffer.wrap(buffer);
        FileChannel channel = in.getChannel();
        try {
            int read;
            for (int readSum = 0; readSum < buffer.length; readSum += read) {
                read = channel.read(byteBuffer);
                if (read >= 0) continue;
                throw new IOException("Reached EOF before entire file could be read!");
            }
        }
        finally {
            FileSystemUtils.close(channel);
            FileSystemUtils.close(in);
        }
        return buffer;
    }

    public static void unjar(File jarFile, File targetDirectory) throws IOException {
        FileSystemUtils.unzip(jarFile, targetDirectory);
    }

    public static void unzip(File zipFile, File targetDirectory) throws IOException {
        FileSystemUtils.unzip(zipFile, targetDirectory, null, null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static List<String> unzip(File zipFile, File targetDirectory, String prefix, Charset charset) throws IOException {
        List<String> list;
        ZipFile zip = null;
        try {
            zip = charset == null ? new ZipFile(zipFile) : new ZipFile(zipFile, charset);
            list = FileSystemUtils.unzip(zip, targetDirectory, prefix);
        }
        catch (Throwable throwable) {
            FileSystemUtils.close(zip);
            throw throwable;
        }
        FileSystemUtils.close(zip);
        return list;
    }

    public static List<String> unzip(ZipFile zip, File targetDirectory, String prefix) throws IOException {
        Enumeration entries = zip.getEntries();
        ArrayList<String> extractedPaths = new ArrayList<String>();
        while (entries.hasMoreElements()) {
            ZipArchiveEntry entry = (ZipArchiveEntry)entries.nextElement();
            if (entry.isDirectory()) continue;
            String filename = entry.getName();
            if (!StringUtils.isEmpty(prefix)) {
                if (!filename.startsWith(prefix)) continue;
                filename = StringUtils.stripPrefix(filename, prefix);
            }
            try (InputStream entryStream = zip.getInputStream(entry);){
                File file = new File(targetDirectory, filename);
                FileSystemUtils.ensureParentDirectoryExists(file);
                try (FileOutputStream outputStream = new FileOutputStream(file);){
                    FileSystemUtils.copy(entryStream, outputStream);
                }
            }
            extractedPaths.add(filename);
        }
        return extractedPaths;
    }

    public static List<String> unzip(InputStream inputStream, File targetDirectory) throws IOException {
        ArrayList<String> extractedPaths = new ArrayList<String>();
        try (ZipInputStream zipStream = new ZipInputStream(inputStream);){
            ZipEntry entry;
            while ((entry = zipStream.getNextEntry()) != null) {
                if (entry.isDirectory()) continue;
                String filename = entry.getName();
                File file = new File(targetDirectory, filename);
                FileSystemUtils.ensureParentDirectoryExists(file);
                try (OutputStream targetStream = Files.newOutputStream(file.toPath(), new OpenOption[0]);){
                    FileSystemUtils.copy(zipStream, targetStream);
                }
                extractedPaths.add(filename);
            }
        }
        return extractedPaths;
    }

    public static void writeLinesUTF8(Path file, Collection<String> lines) throws IOException {
        FileSystemUtils.writeLines(file, lines, StandardCharsets.UTF_8);
    }

    public static void writeLines(Path file, Collection<String> lines, Charset encoding) throws IOException {
        FileSystemUtils.writeFile(file, StringUtils.concat(lines, "\n"), encoding);
    }

    @Deprecated
    public static void writeLines(File file, Collection<String> lines, Charset encoding) throws IOException {
        FileSystemUtils.writeLines(file.toPath(), lines, encoding);
    }

    @Deprecated
    public static void writeFileUTF8(File file, String content) throws IOException {
        FileSystemUtils.writeFileUTF8(file.toPath(), content);
    }

    public static void writeFileUTF8(Path file, String content) throws IOException {
        FileSystemUtils.writeFile(file, content, StandardCharsets.UTF_8);
    }

    public static void writeFile(Path file, String content, Charset encoding) throws IOException {
        FileSystemUtils.ensureParentDirectoryExists(file);
        try (OutputStreamWriter writer = new OutputStreamWriter(Files.newOutputStream(file, new OpenOption[0]), encoding);){
            writer.write(content);
        }
    }

    @Deprecated
    public static void writeFile(File file, String content, Charset encoding) throws IOException {
        FileSystemUtils.writeFile(file.toPath(), content, encoding);
    }

    public static void writeFileBinary(Path file, byte[] bytes) throws IOException {
        FileSystemUtils.ensureParentDirectoryExists(file);
        Files.write(file, bytes, new OpenOption[0]);
    }

    @Deprecated
    public static void writeFileBinary(File file, byte[] bytes) throws IOException {
        FileSystemUtils.writeFileBinary(file.toPath(), bytes);
    }

    public static String readStreamUTF8(InputStream input) throws IOException {
        return FileSystemUtils.readStream(input, StandardCharsets.UTF_8);
    }

    public static String readStream(InputStream input, Charset encoding) throws IOException {
        int n;
        StringBuilder out = new StringBuilder();
        Reader r = FileSystemUtils.streamReader(input, encoding);
        char[] b = new char[4096];
        while ((n = r.read(b)) != -1) {
            out.append(b, 0, n);
        }
        return out.toString();
    }

    public static byte[] readStreamBinary(InputStream input) throws IOException {
        return input.readAllBytes();
    }

    public static Reader streamReader(InputStream in, Charset encoding) throws IOException {
        if (!in.markSupported()) {
            in = new BufferedInputStream(in);
        }
        in.mark(4);
        byte[] prefix = new byte[4];
        EByteOrderMark bom = null;
        try {
            FileSystemUtils.safeRead(in, prefix);
            bom = EByteOrderMark.determineBOM(prefix).orElse(null);
        }
        catch (IOException iOException) {
            // empty catch block
        }
        in.reset();
        if (bom != null) {
            encoding = bom.getEncoding();
            for (int i = 0; i < bom.getBOMLength(); ++i) {
                int n = in.read();
            }
        }
        return new InputStreamReader(in, encoding);
    }

    public static Properties readProperties(Path propertiesFile) throws IOException {
        return FileSystemUtils.readProperties(() -> Files.newInputStream(propertiesFile, new OpenOption[0]));
    }

    private static Properties readProperties(SupplierWithException<? extends InputStream, IOException> streamSupplier) throws IOException {
        try (InputStream stream = streamSupplier.get();){
            Properties props = new Properties();
            props.load(stream);
            Properties properties = props;
            return properties;
        }
    }

    public static File commonRoot(Iterable<? extends File> files) throws IOException {
        HashSet<String> absolutePaths = new HashSet<String>();
        for (File file : files) {
            absolutePaths.add(file.getCanonicalPath());
        }
        CCSMAssert.isTrue(absolutePaths.size() >= 2, "Expected are at least 2 files");
        String longestCommonPrefix = StringUtils.longestCommonPrefix(absolutePaths);
        int n = longestCommonPrefix.lastIndexOf(File.separator);
        if (n > -1) {
            longestCommonPrefix = longestCommonPrefix.substring(0, n);
        }
        if (StringUtils.isEmpty(longestCommonPrefix)) {
            return null;
        }
        return new File(longestCommonPrefix);
    }

    public static InputStream autoDecompressStream(InputStream in) throws IOException {
        if (!in.markSupported()) {
            in = new BufferedInputStream(in);
        }
        in.mark(2);
        boolean isGZIP = (in.read() & 0xFF | (in.read() & 0xFF) << 8) == 35615;
        in.reset();
        if (isGZIP) {
            return new GZIPInputStream(in);
        }
        return in;
    }

    public static void close(ZipFile zipFile) {
        if (zipFile == null) {
            return;
        }
        try {
            zipFile.close();
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    public static void close(Closeable closeable) {
        if (closeable == null) {
            return;
        }
        try {
            closeable.close();
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    public static void sort(List<File> files) {
        files.sort(new FilenameComparator());
    }

    public static String normalizeSeparators(String path) {
        return path.replace(File.separatorChar, '/');
    }

    public static String normalizeSeparatorsPlatformIndependently(String path) {
        if (path.contains(String.valueOf('\\')) && !path.contains(String.valueOf('/'))) {
            return path.replace('\\', '/');
        }
        return path;
    }

    public static File extractJarFileFromJarURL(URL url) {
        CCSMAssert.isTrue("jar".equals(url.getProtocol()), "May only be used with 'jar' URLs!");
        String path = url.getPath();
        CCSMAssert.isTrue(path.startsWith("file:"), "May only be used for URLs pointing to files");
        int index = path.indexOf(33);
        CCSMAssert.isTrue(index >= 0, "Unknown format for jar URLs");
        path = path.substring(0, index);
        return FileSystemUtils.fromURL(path);
    }

    private static File fromURL(String url) {
        url = url.replace(" ", "%20");
        try {
            return new File(new URI(url));
        }
        catch (URISyntaxException e) {
            throw new AssertionError("The assumption is that this method is capable of working with non-standard-compliant URLs, too. Apparently it is not. Invalid URL: " + url + ". Ex: " + e.getMessage(), e);
        }
    }

    public static boolean isAbsolutePath(String filename) {
        if (filename.startsWith("/") || filename.startsWith("~")) {
            return true;
        }
        if (filename.length() > 2 && Character.isLetter(filename.charAt(0)) && filename.charAt(1) == ':') {
            return true;
        }
        return filename.startsWith("\\\\");
    }

    public static void safeRead(InputStream in, byte[] data) throws IOException {
        int read;
        int offset = 0;
        for (int length = data.length; length > 0; length -= read) {
            read = in.read(data, offset, length);
            if (read < 0) {
                throw new EOFException("Reached end of file before completing read.");
            }
            offset += read;
        }
    }

    @Deprecated
    public static File getTmpDir() {
        return new File(TEMP_DIR_PATH);
    }

    public static File getUserHomeDir() {
        return new File(System.getProperty("user.home"));
    }

    public static File getJvmWorkingDirOrTempForDevMode() {
        if (FileSystemUtils.isDevModeOrJunitTest()) {
            return FileSystemUtils.getTmpDir();
        }
        return new File(System.getProperty("user.dir"));
    }

    public static boolean isDevModeOrJunitTest() {
        return Boolean.getBoolean("com.teamscale.dev-mode") || FileSystemUtils.isJUnitTest();
    }

    private static boolean isJUnitTest() {
        StackTraceElement[] stackTrace;
        for (StackTraceElement element : stackTrace = Thread.currentThread().getStackTrace()) {
            if (!element.getClassName().startsWith("org.junit.")) continue;
            return true;
        }
        return false;
    }

    public static boolean isReadableFile(@Nullable Path path) {
        return path != null && Files.isRegularFile(path, new LinkOption[0]) && Files.isReadable(path);
    }

    public static String concatenatePaths(String firstParent, String ... paths) {
        return FileSystemUtils.normalizeSeparators(Paths.get(firstParent, paths).toString());
    }

    public static void recursivelyRemoveDirectoryIfEmpty(File path) throws IOException {
        String[] children = path.list();
        if (children == null) {
            if (path.exists()) {
                return;
            }
        } else if (children.length == 0) {
            FileSystemUtils.deleteFile(path);
        } else {
            return;
        }
        FileSystemUtils.recursivelyRemoveDirectoryIfEmpty(path.getParentFile());
    }

    public static long parseDataSize(String dataSize) {
        String dataSizeWithoutComma = dataSize.replaceAll(",", "");
        int unitBeginIndex = StringUtils.indexOfMatch(dataSizeWithoutComma, DATA_SIZE_UNIT_START_PATTERN);
        if (unitBeginIndex == -1) {
            return Long.parseLong(dataSizeWithoutComma);
        }
        double rawDataSize = Double.parseDouble(dataSizeWithoutComma.substring(0, unitBeginIndex));
        String unitString = dataSizeWithoutComma.substring(unitBeginIndex);
        char unitChar = unitString.charAt(0);
        int power = METRIC_SYSTEM_PREFIXES.indexOf(unitChar) + 1;
        boolean isBinaryPrefix = unitString.length() >= 2 && unitString.charAt(1) == 'i';
        int factor = 1000;
        if (isBinaryPrefix) {
            factor = 1024;
            if (StringUtils.stripSuffix(unitString, "B").length() != 2) {
                throw new NumberFormatException("Malformed data size: " + dataSizeWithoutComma);
            }
        } else if (power == 0 ? !StringUtils.stripSuffix(unitString, "B").isEmpty() : StringUtils.stripSuffix(unitString, "B").length() != 1) {
            throw new NumberFormatException("Malformed data size: " + dataSizeWithoutComma);
        }
        return (long)(rawDataSize * Math.pow(factor, power));
    }

    public static long getLastModifiedTimestamp(File file) throws IOException {
        return Files.getLastModifiedTime(Paths.get(file.toURI()), new LinkOption[0]).toMillis();
    }

    public static String toSafeFilename(String name) {
        name = StringUtils.replaceAll(name, Patterns.MULTIPLE_NON_WORD_CHARACTERS, "-");
        name = StringUtils.replaceAll(name, FILENAME_PLACEHOLDERS, "-");
        return name;
    }

    public static String toValidFileName(String name) {
        return name.replaceAll("[:\\\\/*\"?|<>']", "-");
    }

    public static String readFileSystemIndependent(File file) throws IOException {
        return StringUtils.normalizeLineSeparatorsPlatformIndependent(FileSystemUtils.readFileUTF8(file));
    }

    public static String replaceFilePathFilenameWith(String uniformPath, String newFileName) {
        int folderSepIndex = uniformPath.lastIndexOf(47);
        if (uniformPath.endsWith("/")) {
            return uniformPath + newFileName;
        }
        if (folderSepIndex == -1) {
            return newFileName;
        }
        return uniformPath.substring(0, folderSepIndex) + "/" + newFileName;
    }

    public static long calculateDirectorySize(@NonNull Path folder) {
        return FileSystemUtils.calculateDirectorySize(folder, null);
    }

    public static long calculateDirectorySize(@NonNull Path folder, @Nullable Consumer<? super PairList<Path, IOException>> failedFilesConsumer) {
        CCSMAssert.isNotNull((Object)folder, () -> String.format("Expected \"%s\" to be not null", "folder"));
        if (!Files.isDirectory(folder, new LinkOption[0])) {
            throw new IllegalArgumentException(String.format("Provided file \"%s\" is not a directory", folder.toAbsolutePath()));
        }
        LongAdder directorySize = new LongAdder();
        NonThrowingFileVisitor<Path> visitor = new NonThrowingFileVisitor<Path>(file -> {
            BasicFileAttributes fileAttributes = Files.readAttributes(file, BasicFileAttributes.class, new LinkOption[0]);
            if (fileAttributes.isRegularFile()) {
                directorySize.add(fileAttributes.size());
            }
        });
        try {
            Files.walkFileTree(folder, visitor);
        }
        catch (IOException e) {
            CCSMAssert.fail(String.format("Unexpected IOException occurred while calculating directory size of: %s", folder.toAbsolutePath()), e);
        }
        if (failedFilesConsumer != null && !visitor.getFailedFiles().isEmpty()) {
            failedFilesConsumer.accept(visitor.getFailedFiles());
        }
        return directorySize.longValue();
    }

    public static boolean isSystemFileName(String entryName) {
        return entryName.startsWith("__MACOSX") || entryName.endsWith(".DS_Store") || entryName.endsWith("~");
    }

    public static TemporaryDirectory getTemporaryDirectory(String prefix) throws IOException {
        return new TemporaryDirectory(Files.createTempDirectory(prefix, new FileAttribute[0]), false);
    }

    public static TemporaryDirectory getTemporaryDirectoryDeletedOnShutdown(String prefix) throws IOException {
        return new TemporaryDirectory(Files.createTempDirectory(prefix, new FileAttribute[0]), true);
    }

    public static void createFileRelativeToBaseDirectory(String baseDirectory, String uniformPath, String content) throws IOException {
        Path path = Path.of(baseDirectory, uniformPath);
        Files.createDirectories(Objects.requireNonNull(path.getParent()), new FileAttribute[0]);
        FileSystemUtils.writeFileUTF8(path, content);
    }

    private FileSystemUtils() {
        throw new UnsupportedOperationException("Avoid initialization of util class.");
    }
}

