<h2>Why is this an issue?</h2>
<p>Mockito provides <em>argument matchers</em> and <em>argument captors</em> for flexibly stubbing or verifying method calls.</p>
<p><code>Mockito.verify()</code>, <code>Mockito.when()</code>, <code>Stubber.when()</code> and <code>BDDMockito.given()</code> each have overloads
with and without argument matchers.</p>
<p>However, if argument matchers or captors are used only on some of the parameters, all the parameters need to have matchers as well, otherwise an
<code>InvalidUseOfMatchersException</code> will be thrown.</p>
<p>This rule consequently raises an issue every time matchers are not used on all the parameters of a stubbed/verified method.</p>
<h2>How to fix it</h2>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```java
@Test
public void myTest() {
    // Setting up mock responses
    given(foo.bar(anyInt(), i1, i2)).willReturn(null); // Noncompliant, no matchers for "i1" and "i2"
    when(foo.baz(eq(val1), val2)).thenReturn("hi"); // Noncompliant, no matcher for "val2"

    // Simulating exceptions
    doThrow(new RuntimeException()).when(foo).quux(intThat(x -> x >= 42), -1); // Noncompliant, no matcher for "-1"

    // Verifying method invocations
    verify(foo).bar(i1, anyInt(), i2); // Noncompliant, no matchers for "i1" and "i2"

    // Capturing arguments for verification
    ArgumentCaptor<Integer> captor = ArgumentCaptor.forClass(Integer.class);
    verify(foo).bar(captor.capture(), i1, any()); // Noncompliant, no matchers for "i1"
}
```
<h4>Compliant solution</h4>

```java
@Test
public void myTest() {
    // Setting up mock responses
    given(foo.bar(anyInt(), eq(i1), eq(i2))).willReturn(null); // Compliant, all arguments have matchers
    when(foo.baz(val1, val2)).thenReturn("hi"); // Compliant, no argument has matchers

    // Simulating exceptions
    doThrow(new RuntimeException()).when(foo).quux(intThat(x -> x >= 42), eq(-1)); // Compliant, all arguments have matchers

    // Verifying method invocations
    verify(foo).bar(eq(i1), anyInt(), eq(i2)); // Compliant, all arguments have matchers

    // Capturing arguments for verification
    ArgumentCaptor<Integer> captor = ArgumentCaptor.forClass(Integer.class);
    verify(foo).bar(captor.capture(), any(), any()); // Compliant, all arguments have matchers
}
```
<h2>Resources</h2>
<ul>
  <li> <a href="https://javadoc.io/doc/org.mockito/mockito-core/latest/org/mockito/Mockito.html#argument_matchers">Mockito documentation</a> -
  argument matchers </li>
  <li> <a href="https://sonarsource.github.io/rspec/#/rspec/<a href='checks/checks/?languages=JAVA&tools=SONAR_LINT&rule=java:S6068'>java:S6068</a>/java"><a href='checks/checks/?languages=JAVA&tools=SONAR_LINT&rule=java:S6068'>java:S6068</a> - Call to Mockito method "verify", "when" or
  "given" should be simplified</a> </li>
</ul>