<h2>Why is this an issue?</h2>
<p>Kotlin coroutines follow the principle of structured concurrency. This helps in preventing resource leaks and ensures that scopes are only exited
once all child coroutines have exited. Hence, structured concurrency enables developers to build concurrent applications while having to worry less
about cleaning up concurrent tasks manually.</p>
<p>It is possible to break this concept of structured concurrency in various ways. Generally, this is not advised, as it can open the door to
coroutines being leaked or lost. Ask yourself if breaking structured concurrency here is really necessary for the application’s business logic, or if
it could be avoided by refactoring parts of the code.</p>
<p>This rule raises an issue when it detects that the structured concurrency principles are violated. It avoids reporting on valid use cases and in
situations where developers have consciously opted into using delicate APIs (e.g. by using the <code>@OptIn</code> annotation) and hence should be
aware of the possible pitfalls.</p>
<h3>Noncompliant code example</h3>
<p><a href="https://kotlin.github.io/kotlinx.coroutines/kotlinx-coroutines-core/kotlinx.coroutines/-global-scope/index.html">GlobalScope</a>:</p>

```kotlin
fun main() {
  GlobalScope.launch { // Noncompliant: no explicit opt-in to DelicateCoroutinesApi
    // Do some work
  }.join()
}
```
<p>Manual job instantiation:</p>

```kotlin
fun startLongRunningBackgroundJob(job: Job) {
    val coroutineScope = CoroutineScope(job)
    coroutineScope.launch(Job()) { // Noncompliant: new job instance passed to launch()
        // Do some work
    }
}
```
<p>Manual supervisor instantiation:</p>

```kotlin
    coroutineScope {
        launch(SupervisorJob()) { // Noncompliant: new supervisor instance passed to launch()
            // Do some work
        }
    }
```
<h3>Compliant solution</h3>
<p>In many situations, a good pattern is to use <code>coroutineScope</code> as provided in suspending functions:</p>

```kotlin
suspend fun main() {
    worker()
}

suspend fun worker() {
    coroutineScope {
        launch { // Compliant: no manually created job/supervisor instance passed to launch()
            // Do some work
        }
    }
}
```
<p><a href="https://kotlin.github.io/kotlinx.coroutines/kotlinx-coroutines-core/kotlinx.coroutines/-global-scope/index.html">GlobalScope</a>:</p>

```kotlin
@OptIn(DelicateCoroutinesApi::class)
fun main() {
    GlobalScope.launch { // Compliant: explicit opt-in to DelicateCoroutinesApi via method annotation
        // Do some work
    }.join()
}
```
<p>No manual job instantiation:</p>

```kotlin
fun startLongRunningBackgroundJob(job: Job) {
    val coroutineScope = CoroutineScope(job)
    coroutineScope.launch { // Compliant: no manually created job/supervisor instance passed to launch()
        // Do some work
    }
}
```
<p>Using a supervisor scope instead of manually instantiating a supervisor:</p>

```kotlin
  supervisorScope {
    launch {
      // Do some work
    }
  }
```
<h2>Resources</h2>
<ul>
  <li> <a href="https://kotlinlang.org/docs/coroutines-basics.html#structured-concurrency">Structured concurrency</a> in the Kotlin docs </li>
  <li> <a href="https://kotlin.github.io/kotlinx.coroutines/kotlinx-coroutines-core/kotlinx.coroutines/-global-scope/index.html">GlobalScope
  documentation</a> </li>
  <li> <a href="https://kotlin.github.io/kotlinx.coroutines/kotlinx-coroutines-core/kotlinx.coroutines/-coroutine-scope/index.html">coroutineScope
  documentation</a> </li>
  <li> <a href="https://developer.android.com/kotlin/coroutines/coroutines-best-practices">Android coroutines best practices</a> </li>
</ul>