<h2>Why is this an issue?</h2>
<p>A type guard is a TypeScript feature that allows you to narrow the type of a variable within a conditional block of code. It is a way to tell the
TypeScript compiler that an expression is of a certain type, based on some condition that you check at runtime.</p>

```ts
function printLength(x: any) {
  if (typeof x === 'string') {
    console.log(x.length);
  }
}
```
<p>Type predicates are user-defined functions that work as type guards, where you define yourself how to narrow the type of an expression based on
some custom condition. These are just functions with a return type of <code>argumentName is SomeType</code>. Such functions return <code>true</code>
if the argument is of the specified type.</p>

```ts
function isString(x: any): x is string {
  return typeof x === 'string';
}

function printLength(x: any) {
  if (isString(x)) {
    console.log(x.length);
  }
}
```
<p>One of the advantages of using such a function is that in a conditional block where the condition is a type guard, the compiler automatically
performs the appropriate casts, so explicit casting becomes unnecessary.</p>
<p>Type predicates provide a more precise, readable, and flexible way to check types in your code, which can lead to more robust and maintainable
code.</p>
<p>This rule raises an issue when a boolean function checking for the type of its only argument can be replaced with a type predicate.</p>

```ts
function isSomething(x: BaseType): boolean { // Noncompliant
  return (<Something>x).foo !== undefined;
}

if (isSomething(v)) {
  (<Something>v).foo();
  (v as Something).foo();
}
```
<p>Transform the function into a type predicate, adding a return type annotation that specifies the type predicate.</p>

```ts
function isSomething(x: BaseType): x is Something {
  return (<Something>x).foo !== undefined;
}

if (isSomething(v)) {
  v.foo();
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> <a href="https://www.typescriptlang.org/docs/handbook/2/narrowing.html#using-type-predicates">TypeScript - Using type predicates</a> </li>
</ul>