Suppressing warnings can hide potential issues in the code. Try to fix the underlying issue and use the most specific warning suppression possible.

# What Does This Check Look For?

This check identifies instances in the code where the `@SuppressWarnings` annotation is used. Depending on configuration all usages or only usages that suppress specific warnings are flagged.

# Why is This a Problem?

Using `@SuppressWarnings` can lead to the accumulation of technical debt and mask real problems in the code that should be resolved. It might obscure the visibility of warnings that could point to potential defects, deprecated usage of code, or other important issues that require attention. Ignoring these warnings can degrade code quality over time and make maintenance more difficult.

# How Can I Resolve This?

Review the reasons for suppressing warnings in each case. If possible, refactor the code to resolve the issues rather than suppress warnings. In cases where warnings must be suppressed, document the reason clearly in the code to maintain comprehensibility. Suppress the most specific warnings possible to avoid accidentally masking other issues.

# Examples

## Example 1

The following example assumes the check is configured to deny all usages of `@SuppressWarnings`.

### Non-Compliant

```java
public class Test {
    @SuppressWarnings("unused") // non-compliant
    private int counter;

    public void foo() {
        // The counter variable is not used, so it would normally
        // cause a warning. However, the @SuppressWarnings("unused") annotation
        // suppresses this warning.
    }
}
```

### Compliant

```java
public class Test {
    private int counter;

    public int foo() {
      // ...
      return counter + 1;
    }
}
```

## Example 2

The following example assumes the check is configured to allow some usages of `@SuppressWarnings` but not `@SuppressWarnings("all")`.

### Non-Compliant

```java
public class Test {

	<E> void test() {
		@SuppressWarnings("all")
		E[] arr = (E[]) new Object[1];
	}
}
```

### Compliant

```java
public class Test {

	<E> void test() {
		@SuppressWarnings("unchecked")
		E[] arr = (E[]) new Object[1];
	}
}
```
