In some codebases, developers might want to avoid certain function calls because they may introduce unwanted side effects, reduce performance, or skip best practices.

# What Does This Check Look For?
This check searches the code for unwanted function calls listed in the analysis profile.
By default, this check does not come with a pre-defined list of unwanted functions.

To specify the functions that should be detected by this check, the user has to specify a list of function names.
These function names may either be standalone names for local or global functions (e.g., `printf` or `std::getline`), or fully-qualified static member functions (e.g., `MyClass::foo`)

# Why is This a Problem?
Generally speaking, some function calls are discouraged because they might not have the effect the developer wants.
For instance, function calls left in the code for debugging purposes or for skipping tests.

# How Can I Resolve This?
It depends on the unwanted function. 
In some cases, the fix might be just the removal of the function call, while in some other cases the developers might have an agreed solution for each unwanted function call listed in the project configuration.

# Limitation
* This check does not detect calls to member functions on object instances (e.g., `myClassInstance.foo()` if `MyClass.foo` is configured as unwanted function).
* This check does not consider function calls via dynamic call constructs, like function pointers or `std::function`.

# Examples
## Non-Compliant (C)
*Unwanted function calls* configuration: `printf`
```C
#include <stdio.h>
#include <stdlib.h>
#include <time.h>

int main () {
   int i, n;
   time_t t;
   
   srand((unsigned) time(&t));
   
   char word[6] = "banana"; 

    n = rand() % 5;
    
    /* Non-compliant call to printf() assuming the project doesn't allow printf */
   for( i = 0 ; i <= n ; i++ ) {
      printf("%s\n", word);
   }
   
   return(0);
}
```

## Compliant (C)
```c
#include <stdio.h>
#include <stdlib.h>
#include <time.h>

int main () {
   int i, n;
   time_t t;
   
   srand((unsigned) time(&t));
   
   char word[6] = "banana"; 

    n = rand() % 5;
    
    /* Compliant call to puts() replacing printf() */
   for( i = 0 ; i <= n ; i++ ) {
      puts(word);
   }
   
   return(0);
}
```

## Non-Compliant (Swift)
*Unwanted function calls* configuration: `XCTSkip`
```swift
/* Non-compliant call to unwanted XCTSkip() */
func test_skipMe() throws {
   throw XCTSkip("Skipping this test")
   // ...
}
```

## Compliant (Swift)

```swift
/* Compliant call to XCTSkipIf replacing XCTSkip() */
func test_skipIf() throws {
   try XCTSkipIf(someConditionIsTrue(), "Skipping this test")
   // ...
}

// OR

/* Compliant call to XCTSkipUnless replacing XCTSkip() */
func test_skipUnless() throws {
   try XCTSkipUnless(someConditionIsFalse(), "Skipping this test")
   // ...
}
```