### SCS0029 - Cross-Site Scripting (XSS) 

A potential XSS was found. The endpoint returns a variable from the client input that has not been encoded. To protect against stored XSS attacks, make sure any dynamic content coming from user or data store cannot be used to inject JavaScript on a page. Most modern frameworks will escape dynamic content by default automatically (Razor for example) or by using special syntax (`<%: content %>`, `<%= HttpUtility.HtmlEncode(content) %>`).

#### Risk

XSS could be used to execute unwanted JavaScript in a client’s browser. XSS can be used to steal the cookie containing the user’s session ID. There is rarely a good reason to read or manipulate cookies in client-side JavaScript, so consider marking cookies as [HTTP-only](#SCS0009), meaning that cookies will be received, stored, and sent by the browser, but cannot be modified or read by JavaScript.

#### Vulnerable Code

```cs
public class TestController : Controller
{
    [HttpGet(""{myParam}"")]
    public string Get(string myParam)
    {
        return "value " + myParam;
    }
}

```

#### Solution

```cs
public class TestController : Controller
{
    [HttpGet(""{myParam}"")]
    public string Get(string myParam)
    {
        return "value " + HttpUtility.HtmlEncode(myParam);
    }
}

```

#### References

[CWE-79: Improper Neutralization of Input During Web Page Generation (‘Cross-site Scripting’)](https://cwe.mitre.org/data/definitions/79.html)  
[WASC-8: Cross Site Scripting](http://projects.webappsec.org/w/page/13246920/Cross%20Site%20Scripting)  
[OWASP: XSS Prevention Cheat Sheet](https://www.owasp.org/index.php/XSS%5F%28Cross%5FSite%5FScripting%29%5FPrevention%5FCheat%5FSheet)  
[OWASP: Top 10 2013-A3: Cross-Site Scripting (XSS)](https://www.owasp.org/index.php/Top%5F10%5F2013-A3-Cross-Site%5FScripting%5F%28XSS%29)

Source: [Security Code Scan](https://security-code-scan.github.io/#Rules)