<h2>Why is this an issue?</h2>
<p>When overloading some arithmetic operator overloads, it is very important to make sure that all related operators and methods are consistent in
their implementation.</p>
<p>The following guidelines should be followed:</p>
<ul>
  <li> When providing <code>operator ==, !=</code> you should also provide <code>Equals(Object)</code> and <code>GetHashCode()</code>. </li>
  <li> When providing <code>operator +, -, *, / or %</code>&nbsp;you should also provide <code>operator ==</code>, respecting the previous guideline.
  </li>
</ul>
<p>This rule raises an issue when any of these guidelines are not followed on a publicly-visible class or struct (<code>public</code>,
<code>protected</code> or <code>protected internal</code>).</p>
<h2>How to fix it</h2>
<p>Make sure to implement all related operators.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>
<pre data-diff-id="1" data-diff-type="noncompliant">
public class Foo // Noncompliant
{
    private int left;
    private int right;

    public Foo(int l, int r)
    {
        this.left = l;
        this.right = r;
    }

    public static Foo operator +(Foo a, Foo b)
    {
        return new Foo(a.left + b.left, a.right + b.right);
    }

    public static Foo operator -(Foo a, Foo b)
    {
        return new Foo(a.left - b.left, a.right - b.right);
    }
}
</pre>
<h4>Compliant solution</h4>
<pre data-diff-id="1" data-diff-type="compliant">
public class Foo
{
    private int left;
    private int right;

    public Foo(int l, int r)
    {
      this.left = l;
      this.right = r;
    }

    public override bool Equals(Object obj)
    {
        var a = obj as Foo;
        if (a == null)
          return false;
        return this == a;
    }

    public override int GetHashCode()
    {
        return HashCode.Combine(right, left);
    }

    public static Foo operator +(Foo a, Foo b)
    {
        return new Foo(a.left + b.left, a.right + b.right);
    }

    public static Foo operator -(Foo a, Foo b)
    {
        return new Foo(a.left - b.left, a.right - b.right);
    }

    public static bool operator ==(Foo a, Foo b)
    {
        return a.left == b.left &amp;&amp; a.right == b.right;
    }

    public static bool operator !=(Foo a, Foo b)
    {
        return !(a == b);
    }
}
</pre>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> Microsoft Learn - <a href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/operator-overloading">Operator
  overloading</a> </li>
  <li> Microsoft Learn - <a href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/equality-operators">Equality
  operators</a> </li>
  <li> Microsoft Learn - <a href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/arithmetic-operators">Arithmetic
  operators (C# reference)</a> </li>
</ul>

