<h2>Why is this an issue?</h2>
<p>When using regular expressions, a capturing group provides extra information about the matched pattern. Named groups will store the matched
contents on the <code>groups</code> property of the returned matches.</p>

```js
const regex = /(?<month>[0-9]{2})\/(?<year>[0-9]{2})/;
const { groups: {month, year} } = regex.exec("01/02"); // month is '01', year is '02'
```
<p>This rule raises issues in several cases:</p>
<ul>
  <li> Named groups are defined but never called by their name. To make the code more readable, remove unused named capturing groups. </li>
</ul>

```js
const score = "14:1";
const scorePattern = /(?<player1>[0-9]+):(?<player2>[0-9]+)/; // Noncompliant - named groups are never used

if (scorePattern.exec(score)) {
  checkScore(score);
}
```
<ul>
  <li> Named groups are defined but called by their index. To make the code more readable:
    <ul>
      <li> access the matched contents using the named group; or </li>
      <li> remove it from the pattern. </li>
    </ul>  </li>
</ul>

```js
const datePattern = /(?<month>[0-9]{2})/(?<year>[0-9]{2})/;
const dateMatcher = datePattern.exec("01/02");

if (dateMatcher) {
  checkValidity(dateMatcher[1], dateMatcher[2]);  // Noncompliant - group indexes are used instead of names
}
```
<ul>
  <li> Named groups are referenced while not defined. This can have undesired effects, as it will return <code>undefined</code> if there are other
  named groups. If there are none, <code>groups</code> will be <code>undefined</code>, and trying to access the named group will throw a
  <code>TypeError</code> if its existence has not been checked. </li>
</ul>

```js
const datePattern = /(?<month>[0-9]{2})/(?<year>[0-9]{2})/;
const dateMatcher = datePattern.exec("01/02");

if (dateMatcher) {
  checkValidity(dateMatcher.groups.day); // Noncompliant - there is no group called "day", returns `undefined`
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Guide/Regular_expressions">Regular expressions</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Guide/Regular_expressions/Groups_and_backreferences">Groups and
  backreferences</a> </li>
</ul>