Long functions are more difficult to understand, document, reuse and test.
Extract appropriate helper functions to reduce the length.

# What Does This Check Look For?

This check counts the length of methods, functions or other subroutines.
The length is counted, depending on the configuration in the analysis profile, in Source Lines of Code (SLOC), i.e. all lines that are not empty or comments, or in statements.

Depending on the analysis profile configuration, this check produces yellow findings for lengthy functions (default: longer than 30 SLOC) and red findings for very long functions (default: longer than 75 SLOC).
It is also possible to configure this check to count by statements rather than source lines of code.

# Why is This a Problem?

Long functions are an issue for several related, but distinct reasons:

1. **Comprehensibility.**
Long functions tend to make the code harder to understand because each line adds more context that you need to keep in your mind.
This also makes them more difficult to modify, because you will often need to fully understand it before you can modify it safely.
Shorter functions, on the other hand, have a smaller scope and less context to keep in mind and are, thus, easier to understand and modify.
As a result, shorter functions are typically less error-prone (see references).
Furthermore, elevating code into a method with a descriptive name enhances readability and maintainability by clearly and concisely conveying the purpose and functionality of the method. Additionally, this approach can help improve code organization and facilitate future changes.

2. **Reuse.**
Long functions often lead to reuse of code fragments through copy-and-paste.
Breaking them down into shorter functions, allows them to be reused without introducing duplication and makes the code more modular.

3. **Testing.**
The functionality in long functions is harder to test deliberately.

# How Can I Resolve This?

The best way to reduce the length is typically the extraction of smaller helper functions with speaking names.
This adds a layer of abstraction for cohesive parts of the code and keeps the context smaller and, thus, easier to understand.
Often, inline comments already hint at good points for splitting a sequence of statements and can give an inspiration for the name of the extracted method.

# Examples
## Non-Compliant

This example uses Java, but the principle generalizes to other languages.

```java
public static int loadHighestLeveragedPosition(String inputFilePath) throws CustomFileAccessException {
    if (inputFilePath == null || inputFilePath.isEmpty()) {
        throw new IllegalArgumentException("The input path may not be empty nor null!");
    }

    // Load file content
    String fileContent;
    try {
        File file = new File(inputFilePath);
        fileContent = new String(Files.readAllBytes(file.toPath()));
    } catch (IOException e) {
        LOGGER.error("Can not read from input file!", e);
        throw new CustomFileAccessException(e);
    }

    String[] lines = fileContent.split("\n");
    Set<Integer> registeredPositions = new HashSet<>();

    for (String line : lines) {
        // Extract position values signaled by opening angle bracket
        if (line.startsWith("<")) {
            String matchingString = POSITION_REGEX.matcher(line).group(POSITION_GROUP_LABEL);
            if (matchingString == null) {
                continue;
            }

            int position = Integer.parseInt(matchingString);
            if (position <= 42) {
                // position is too small, do not register
                continue;
            }

            registeredPositions.add(position);
        }
    }

    // Calculate the leveraged positions i.e. the ones, which are divisible by 4.
    Set<Integer> leveragedPositions = new HashSet<>();
    for (int position : registeredPositions) {
        if (position % 4 == 0) {
            leveragedPositions.add(position);
        }
    }

    return leveragedPositions.stream().max(Integer::compare).orElse(0);
}
```

## Compliant

```java
public static int loadHighestLeveragedPosition(String inputFilePath) throws CustomFileAccessException {
    String fileContent = loadFileContent(inputFilePath);

    String[] lines = fileContent.split("\n");
    Set<Integer> registeredPositions = new HashSet<>();

    for (String line : lines) {
        Integer position = extractPosition(line);
        if (position != null) {
            registeredPositions.add(position);
        }
    }

    return calculateHighestLeveragedPosition(registeredPositions);
}

private static String loadFileContent(String inputFilePath) throws CustomFileAccessException {
    if (inputFilePath == null || inputFilePath.isEmpty()) {
        throw new IllegalArgumentException("The input path may not be empty nor null!");
    }

    String fileContent;
    try {
        File file = new File(inputFilePath);
        fileContent = new String(Files.readAllBytes(file.toPath()));
    } catch (IOException e) {
        LOGGER.error("Can not read from input file!", e);
        throw new CustomFileAccessException(e);
    }
    return fileContent;
}

/** Calculate the leveraged positions i.e. the ones, which are divisible by 4 and return the highest one */
private static int calculateHighestLeveragedPosition(Set<Integer> registeredPositions) {
    Set<Integer> leveragedPositions = new HashSet<>();
    for (int position : registeredPositions) {
        if (position % 4 == 0) {
            leveragedPositions.add(position);
        }
    }

    return leveragedPositions.stream().max(Integer::compare).orElse(0);
}

/** An opening angle bracket signals a position value */
private static Integer extractPosition(String line) {
    if (!line.startsWith("<")) {
        return null;
    }

    String matchingString = POSITION_REGEX.matcher(line).group(POSITION_GROUP_LABEL);
    if (matchingString == null) {
        return null;
    }

    int position = Integer.parseInt(matchingString);
    if (position <= 42) {
        // position is too small, do not register
        return null;
    }

    return position;
}
```

# Where Can I Learn More?

* [Teamscale Documentation: What Makes Maintainable Code? - Code Structure](https://docs.teamscale.com/introduction/understanding-the-maintainability-of-your-code-base/#code-structure)
* [CQSE Software Quality Blog: The Real Benefits of Short Methods](https://www.cqse.eu/en/news/blog/the-real-benefits-of-short-methods/)
* [Measuring program comprehension: A large-scale field study with professionals](https://xin-xia.github.io/publication/TSE17.pdf)
* [Mining Metrics to Predict Component Failures (Nagappan et al., 2006)](https://www.microsoft.com/en-us/research/wp-content/uploads/2016/02/tr-2005-149.pdf)
* Clean Code: A Handbook of Agile Software Craftsmanship. Chapter 3: Functions (Martin, 2008)
