/*
 * Decompiled with CFR 0.152.
 */
package org.conqat.lib.commons.io;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.time.Duration;
import java.util.Objects;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.function.BiFunction;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.conqat.engine.core.core.ConQATException;
import org.conqat.lib.commons.assertion.CCSMAssert;
import org.conqat.lib.commons.io.StreamConsumingThread;
import org.conqat.lib.commons.string.StringUtils;
import org.conqat.lib.commons.system.SystemUtils;
import org.conqat.lib.commons.utils.UtilsInstantiationNotSupportedException;

public final class ProcessUtils {
    private static final int CONCURRENT_EXTERNAL_PROCESSES = Integer.getInteger("com.teamscale.external-process.maximum-concurrency", Integer.MAX_VALUE);
    private static final Semaphore CONCURRENCY_LIMIT = new Semaphore(CONCURRENT_EXTERNAL_PROCESSES, true);
    private static final Pattern CODE_PAGE_NUMBER_PATTERN = Pattern.compile("\\d+");
    public static final Charset CONSOLE_CHARSET = ProcessUtils.determineConsoleCharset();
    private static final Logger LOGGER = LogManager.getLogger();

    private static Charset determineConsoleCharset() {
        if (SystemUtils.isWindows()) {
            try {
                Charset temporaryCharset = StandardCharsets.UTF_8;
                DefaultStreamConsumer stdoutConsumer = new DefaultStreamConsumer(temporaryCharset, true);
                ProcessUtils.executeWithoutConcurrencyLimit(new ProcessBuilder("chcp.com"), null, -1L, temporaryCharset, stdoutConsumer, new DefaultStreamConsumer(temporaryCharset, false));
                Matcher matcher = CODE_PAGE_NUMBER_PATTERN.matcher(stdoutConsumer.getContent());
                if (matcher.find()) {
                    String charsetName = "Cp" + matcher.group();
                    return Charset.forName(charsetName);
                }
            }
            catch (IOException | IllegalArgumentException exception) {
                // empty catch block
            }
        }
        return StandardCharsets.UTF_8;
    }

    public static ExecutionResult execute(String[] completeArguments) throws IOException {
        return ProcessUtils.execute(completeArguments, null);
    }

    public static ExecutionResult executeInDirectory(String[] completeArguments, File directory) throws IOException {
        ProcessBuilder builder = new ProcessBuilder(completeArguments);
        builder.directory(directory);
        return ProcessUtils.execute(builder, null);
    }

    public static ExecutionResult execute(String[] completeArguments, @Nullable String input) throws IOException {
        ProcessBuilder builder = new ProcessBuilder(completeArguments);
        return ProcessUtils.execute(builder, input);
    }

    public static ExecutionResult execute(ProcessBuilder builder) throws IOException {
        return ProcessUtils.execute(builder, null);
    }

    public static ExecutionResult execute(ProcessBuilder builder, @Nullable String input) throws IOException {
        return ProcessUtils.execute(builder, input, -1L);
    }

    public static ExecutionResult execute(ProcessBuilder builder, @Nullable String input, long timeout) throws IOException {
        return ProcessUtils.execute(builder, input, timeout, true);
    }

    public static ExecutionResult execute(ProcessBuilder builder, @Nullable String input, @Nullable Duration timeout) throws IOException {
        return ProcessUtils.execute(builder, input, Objects.requireNonNullElse(timeout, Duration.ZERO).toSeconds());
    }

    public static ExecutionResult execute(ProcessBuilder builder, @Nullable String input, long timeout, boolean collectOutputStreamContent) throws IOException {
        DefaultStreamConsumer stdoutConsumer = new DefaultStreamConsumer(CONSOLE_CHARSET, collectOutputStreamContent);
        DefaultStreamConsumer stderrConsumer = new DefaultStreamConsumer(CONSOLE_CHARSET, collectOutputStreamContent);
        int exitCode = ProcessUtils.execute(builder, input, timeout, stdoutConsumer, stderrConsumer);
        return new ExecutionResult(stdoutConsumer.getContent(), stderrConsumer.getContent(), exitCode, exitCode == -1);
    }

    public static int execute(ProcessBuilder builder, @Nullable String input, long timeout, IStreamConsumer stdoutConsumer, IStreamConsumer stderrConsumer) throws IOException {
        return ProcessUtils.executeWithConcurrencyLimit(builder, input, timeout, stdoutConsumer, stderrConsumer);
    }

    public static Path getLocationForExecutableOnPath(String executable) throws ConQATException {
        ProcessBuilder builder = new ProcessBuilder(new String[0]);
        if (SystemUtils.isWindows()) {
            builder.command("where", executable);
        } else {
            builder.command("which", executable);
        }
        try {
            ExecutionResult result = ProcessUtils.execute(builder);
            if (result.returnCode == 0) {
                return Path.of(result.getStdout().trim(), new String[0]);
            }
            throw new ConQATException(String.format("Failed to determine location for executable: %s\n%s", executable, result));
        }
        catch (IOException e) {
            throw new ConQATException(String.format("Failed to determine location for executable: %s", executable), e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static int executeWithConcurrencyLimit(ProcessBuilder builder, @Nullable String input, long timeout, IStreamConsumer stdoutConsumer, IStreamConsumer stderrConsumer) throws IOException {
        try {
            CONCURRENCY_LIMIT.acquire();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            return -1;
        }
        try {
            int n = ProcessUtils.executeWithoutConcurrencyLimit(builder, input, timeout, CONSOLE_CHARSET, stdoutConsumer, stderrConsumer);
            return n;
        }
        finally {
            CONCURRENCY_LIMIT.release();
        }
    }

    private static int executeWithoutConcurrencyLimit(ProcessBuilder builder, @Nullable String input, long timeout, Charset consoleCharset, IStreamConsumer stdoutConsumer, IStreamConsumer stderrConsumer) throws IOException {
        Process process = builder.start();
        StreamConsumingThread stderrReader = new StreamConsumingThread(process.getErrorStream(), stderrConsumer);
        StreamConsumingThread stdoutReader = new StreamConsumingThread(process.getInputStream(), stdoutConsumer);
        if (input != null) {
            OutputStreamWriter stdIn = new OutputStreamWriter(process.getOutputStream(), consoleCharset);
            stdIn.write(input);
            ((Writer)stdIn).close();
        }
        try {
            boolean processTimeoutOrInterruption;
            boolean bl = processTimeoutOrInterruption = !ProcessUtils.waitForProcess(process, timeout);
            if (processTimeoutOrInterruption) {
                stderrReader.stopPropagation();
                stdoutReader.stopPropagation();
                return -1;
            }
            stderrReader.join();
            stdoutReader.join();
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
        return process.exitValue();
    }

    private static boolean waitForProcess(Process process, long maxRuntimeSeconds) {
        boolean processFinished;
        try {
            long timeout = maxRuntimeSeconds;
            if (timeout <= 0L) {
                timeout = Long.MAX_VALUE;
            }
            processFinished = process.waitFor(timeout, TimeUnit.SECONDS);
        }
        catch (InterruptedException ignored) {
            processFinished = false;
        }
        if (!processFinished && !ProcessUtils.killProcessTree(process.toHandle())) {
            LOGGER.error("Could not kill process or subprocess of " + process.info().command().orElse(""));
        }
        return processFinished;
    }

    private static boolean killProcessTree(ProcessHandle processHandle) {
        boolean killedAllDescendants = processHandle.children().allMatch(ProcessUtils::killProcessTree);
        boolean isProcessDead = ProcessUtils.killProcess(processHandle);
        return killedAllDescendants && isProcessDead;
    }

    private static boolean killProcess(ProcessHandle processHandle) {
        if (processHandle.supportsNormalTermination()) {
            processHandle.destroy();
            ProcessUtils.pauseThreadFor100ms();
        }
        if (processHandle.isAlive()) {
            processHandle.destroyForcibly();
            ProcessUtils.pauseThreadFor100ms();
        }
        return !processHandle.isAlive();
    }

    private static void pauseThreadFor100ms() {
        try {
            Thread.sleep(100L);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }

    public static <T extends Exception> ExecutionResult executeOrThrow(ProcessBuilder builder, String input, int timeout, BiFunction<String, Throwable, T> exceptionConstructor) throws T {
        String commandString = StringUtils.concat(builder.command(), " ");
        try {
            ExecutionResult result = ProcessUtils.execute(builder, input, timeout);
            if (result.terminatedByTimeoutOrInterruption()) {
                throw (Exception)exceptionConstructor.apply("Process " + commandString + " timed out.", null);
            }
            if (result.getReturnCode() != 0) {
                throw (Exception)exceptionConstructor.apply("Process " + commandString + " failed with non-zero exit code " + result.getReturnCode() + ". Standard output: '" + result.getStdout() + "', Error output: '" + result.getStderr() + "'", null);
            }
            return result;
        }
        catch (IOException e) {
            throw (Exception)exceptionConstructor.apply("Failed to execute " + commandString, e);
        }
    }

    public static DefaultStreamConsumer defaultStreamConsumer(boolean storeContent) {
        return new DefaultStreamConsumer(CONSOLE_CHARSET, storeContent);
    }

    private ProcessUtils() {
        throw new UtilsInstantiationNotSupportedException();
    }

    public static final class DefaultStreamConsumer
    implements IStreamConsumer {
        private final Charset charset;
        private final boolean storeContent;
        private final StringBuilder content = new StringBuilder();

        private DefaultStreamConsumer(@NonNull Charset charset, boolean storeContent) {
            CCSMAssert.isNotNull((Object)charset, () -> String.format("Expected \"%s\" to be not null", "charset"));
            this.storeContent = storeContent;
            this.charset = charset;
        }

        @Override
        public synchronized void consume(InputStream stream) throws IOException {
            BufferedReader reader = new BufferedReader(new InputStreamReader(stream, this.charset));
            char[] buffer = new char[1024];
            int read = reader.read(buffer);
            while (read != -1) {
                if (this.storeContent) {
                    this.content.append(buffer, 0, read);
                }
                read = reader.read(buffer);
            }
        }

        public synchronized String getContent() {
            return this.content.toString();
        }
    }

    public static interface IStreamConsumer {
        public void consume(InputStream var1) throws IOException;
    }

    public static class ExecutionResult {
        private final String stdout;
        private final String stderr;
        private final int returnCode;
        private final boolean processTimeoutOrInterruption;

        public ExecutionResult(String stdout, String stderr, int returnCode, boolean processTimeoutOrInterruption) {
            this.stdout = stdout;
            this.stderr = stderr;
            this.returnCode = returnCode;
            this.processTimeoutOrInterruption = processTimeoutOrInterruption;
        }

        public String getStdout() {
            return Objects.requireNonNullElse(this.stdout, "");
        }

        public String getStderr() {
            return Objects.requireNonNullElse(this.stderr, "");
        }

        public int getReturnCode() {
            return this.returnCode;
        }

        public boolean terminatedByTimeoutOrInterruption() {
            return this.processTimeoutOrInterruption;
        }

        public String toString() {
            return this.toLimitedString(Integer.MAX_VALUE);
        }

        public String toLimitedString(int limit) {
            return "Return code: %d\nTerminated by timeout or interruption: %s\nStd out: %s\nStd err: %s".formatted(this.returnCode, this.processTimeoutOrInterruption, StringUtils.truncateWithEllipsis(this.getStdout(), limit), StringUtils.truncateWithEllipsis(this.getStderr(), limit));
        }
    }
}

