<h2>Why is this an issue?</h2>
<p>In Java, the <code>Object.equals()</code> method is used for object comparison, and it is typically overridden in classes to provide a custom
equality check based on your criteria for equality.</p>
<p>The default implementation of <code>equals()</code> provided by the <code>Object</code> class compares the memory references of the two objects,
that means it checks if the objects are actually the same instance in memory.</p>
<p>The "equals" as a method name should be used exclusively to override <code>Object.equals(Object)</code> to prevent confusion.</p>
<p>It is important to note that when you override <code>equals()</code>, you should also override the <code>hashCode()</code> method to maintain the
contract between <code>equals()</code> and <code>hashCode()</code>.</p>
<h2>How to fix it</h2>
<p>Either override <code>Object.equals(Object)</code> or rename the method.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```java
class MyClass {
  private int foo = 1;

  public boolean equals(MyClass o) {  // Noncompliant; does not override Object.equals(Object)
    return o != null && o.foo == this.foo;
  }

  public static void main(String[] args) {
    MyClass o1 = new MyClass();
    Object o2 = new MyClass();
    System.out.println(o1.equals(o2));  // Prints "false" because o2 an Object not a MyClass
  }
}
```
<h4>Compliant solution</h4>

```java
class MyClass {
  private int foo = 1;

  @Override
  public boolean equals(Object o) {  // Compliant
    if (this == o) {
        return true;
    }

    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    MyClass other = (MyClass)o;
    return this.foo == other.foo;
  }
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> <a href="https://docs.oracle.com/en/java/javase/20/docs/api/java.base/java/lang/Object.html#equals(java.lang.Object)">Oracle SDK -
  Object.equals(Object)</a> </li>
</ul>