<p>Gratuitous boolean expressions are conditions that do not change the evaluation of a program. This issue can indicate logical errors and affect the
correctness of an application, as well as its maintainability.</p>
<h2>Why is this an issue?</h2>
<p>Control flow constructs like <code>if</code>-statements allow the programmer to direct the flow of a program depending on a boolean expression.
However, if the condition is always true or always false, only one of the branches will ever be executed. In that case, the control flow construct and
the condition no longer serve a purpose; they become <em>gratuitous</em>.</p>
<h3>What is the potential impact?</h3>
<p>The presence of gratuitous conditions can indicate a logical error. For example, the programmer <em>intended</em> to have the program branch into
different paths but made a mistake when formulating the branching condition. In this case, this issue might result in a bug and thus affect the
reliability of the application. For instance, it might lead to the computation of incorrect results.</p>
<p>Additionally, gratuitous conditions and control flow constructs introduce unnecessary complexity. The source code becomes harder to understand, and
thus, the application becomes more difficult to maintain.</p>
<h2>How to fix it</h2>
<p>Gratuitous boolean expressions are suspicious and should be carefully removed from the code.</p>
<p>First, the boolean expression in question should be closely inspected for logical errors. If a mistake was made, it can be corrected so the
condition is no longer gratuitous.</p>
<p>If it becomes apparent that the condition is actually unnecessary, it can be removed. The associated control flow construct (e.g., the
<code>if</code>-statement containing the condition) will be adapted or even removed, leaving only the necessary branches.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```java
public class MyClass {
    public void doThings(boolean b, boolean c) {
        a = true;
        if (a) {                // Noncompliant
          doSomething();
        }

        if (b && a) {           // Noncompliant; "a" is always "true"
          doSomething();
        }

        if (c || !a) {          // Noncompliant; "!a" is always "false"
          doSomething();
        }

        if (c || (!c && b)) {   // Noncompliant; c || (!c && b) is equal to c || b
          doSomething();
        }
    }
}
```
<h4>Compliant solution</h4>

```java
public class MyClass {
    public void doThings(boolean b, boolean c) {
        a = true;
        if (foo(a)) {
          doSomething();
        }

        if (b) {
          doSomething();
        }

        if (c) {
          doSomething();
        }

        if (c || b) {
          doSomething();
        }
    }
}
```
<h2>Resources</h2>
<h3>Articles &amp; blog posts</h3>
<ul>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/571">CWE-571 - Expression is Always True</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/570">CWE-570 - Expression is Always False</a> </li>
</ul>