<p>This vulnerability exposes encrypted data to attacks whose goal is to recover the plaintext.</p>
<h2>Why is this an issue?</h2>
<p>Encryption algorithms are essential for protecting sensitive information and ensuring secure communications in a variety of domains. They are used
for several important reasons:</p>
<ul>
  <li> Confidentiality, privacy, and intellectual property protection </li>
  <li> Security during transmission or on storage devices </li>
  <li> Data integrity, general trust, and authentication </li>
</ul>
<p>When selecting encryption algorithms, tools, or combinations, you should also consider two things:</p>
<ol>
  <li> No encryption is unbreakable. </li>
  <li> The strength of an encryption algorithm is usually measured by the effort required to crack it within a reasonable time frame. </li>
</ol>
<p>In today’s cryptography, the length of the <strong>key</strong> directly affects the security level of cryptographic algorithms.</p>
<p>Note that depending on the algorithm, the term <strong>key</strong> refers to a different mathematical property. For example:</p>
<ul>
  <li> For RSA, the key is the product of two large prime numbers, also called the <strong>modulus</strong>. </li>
  <li> For AES and Elliptic Curve Cryptography (ECC), the key is only a sequence of randomly generated bytes.
    <ul>
      <li> In some cases, AES keys are derived from a master key or a passphrase using a Key Derivation Function (KDF) like PBKDF2 (Password-Based Key
      Derivation Function 2) </li>
    </ul>  </li>
</ul>
<p>If an application uses a key that is considered short and <strong>insecure</strong>, the encrypted data is exposed to attacks aimed at getting at
the plaintext.</p>
<p>In general, it is best practice to expect a breach: that a user or organization with malicious intent will perform cryptographic attacks on this
data after obtaining it by other means.</p>
<h3>What is the potential impact?</h3>
<p>After retrieving encrypted data and performing cryptographic attacks on it on a given timeframe, attackers can recover the plaintext that
encryption was supposed to protect.</p>
<p>Depending on the recovered data, the impact may vary.</p>
<p>Below are some real-world scenarios that illustrate the potential impact of an attacker exploiting the vulnerability.</p>
<h4>Additional attack surface</h4>
<p>By modifying the plaintext of the encrypted message, an attacker may be able to trigger additional vulnerabilities in the code. An attacker can
further exploit a system to obtain more information.<br> Encrypted values are often considered trustworthy because it would not be possible for a
third party to modify them under normal circumstances.</p>
<h4>Breach of confidentiality and privacy</h4>
<p>When encrypted data contains personal or sensitive information, its retrieval by an attacker can lead to privacy violations, identity theft,
financial loss, reputational damage, or unauthorized access to confidential systems.</p>
<p>In this scenario, the company, its employees, users, and partners could be seriously affected.</p>
<p>The impact is twofold, as data breaches and exposure of encrypted data can undermine trust in the organization, as customers, clients and
stakeholders may lose confidence in the organization’s ability to protect their sensitive data.</p>
<h4>Legal and compliance issues</h4>
<p>In many industries and locations, there are legal and compliance requirements to protect sensitive data. If encrypted data is compromised and the
plaintext can be recovered, companies face legal consequences, penalties, or violations of privacy laws.</p>
<h2>How to fix it in Java Cryptographic Extension</h2>
<h3>Code examples</h3>
<p>The following code examples either explicitly or implicitly generate keys. Note that there are differences in the size of the keys depending on the
algorithm.</p>
<p>Due to the mathematical properties of the algorithms, the security requirements for the key size vary depending on the algorithm.<br> For example,
a 256-bit ECC key provides about the same level of security as a 3072-bit RSA key and a 128-bit symmetric key.</p>
<h4>Noncompliant code example</h4>
<p>Here is an example of a private key generation with RSA:</p>

```kotlin
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;

fun main(args: Array<String>) {
    try {
        val keyPairGenerator = KeyPairGenerator.getInstance("RSA");
        keyPairGenerator.initialize(1024); // Noncompliant

    } catch (e: NoSuchAlgorithmException) {
        // ...
    }
}
```
<p>Here is an example of a private key generation with AES:</p>

```kotlin
import java.security.KeyGenerator;
import java.security.NoSuchAlgorithmException;

fun main(args: Array<String>) {
    try {
        val keyGenerator = KeyGenerator.getInstance("AES");
        keyGenerator.initialize(64); // Noncompliant

    } catch (e: NoSuchAlgorithmException) {
        // ...
    }
}
```
<p>Here is an example of an Elliptic Curve (EC) initialization. It implicitly generates a private key whose size is indicated in the algorithm
name:</p>

```kotlin
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.security.InvalidAlgorithmParameterException;
import java.security.spec.ECGenParameterSpec;

fun main(args: Array<String>) {
    try {
        val keyPairGenerator  = KeyPairGenerator.getInstance("EC");
        val ellipticCurveName = new ECGenParameterSpec("secp112r1"); // Noncompliant
        keyPairGenerator.initialize(ellipticCurveName);

    } catch (e: NoSuchAlgorithmException) {
        // ...
    } catch (e: InvalidAlgorithmParameterException) {
        // ...
    }
}
```
<h4>Compliant solution</h4>

```kotlin
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;

fun main(args: Array<String>) {
    try {
        val keyPairGenerator = KeyPairGenerator.getInstance("RSA");
        keyPairGenerator.initialize(2048);

    } catch (e: NoSuchAlgorithmException) {
        // ...
    }
}
```

```kotlin
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;

public static void main(String[] args) {
    try {
        val keyPairGenerator = KeyPairGenerator.getInstance("AES");
        keyPairGenerator.initialize(128);

    } catch (e: NoSuchAlgorithmException) {
        // ...
    }
}
```

```kotlin
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.security.InvalidAlgorithmParameterException;
import java.security.spec.ECGenParameterSpec;

public static void main(String[] args) {
    try {
        val keyPairGenerator  = KeyPairGenerator.getInstance("EC");
        val ellipticCurveName = new ECGenParameterSpec("secp256r1");
        keyPairGenerator.initialize(ellipticCurveName);

    } catch (e: NoSuchAlgorithmException) {
        // ...
    } catch (e: InvalidAlgorithmParameterException) {
        // ...
    }
}
```
<h3>How does this work?</h3>
<p>As a rule of thumb, use the cryptographic algorithms and mechanisms that are considered strong by the cryptographic community.</p>
<p>The appropriate choices are the following.</p>
<h4>RSA (Rivest-Shamir-Adleman) and DSA (Digital Signature Algorithm)</h4>
<p>The security of these algorithms depends on the difficulty of attacks attempting to solve their underlying mathematical problem.</p>
<p>In general, a minimum key size of <strong>2048</strong> bits is recommended for both.</p>
<h4>AES (Advanced Encryption Standard)</h4>
<p>AES supports three key sizes: 128 bits, 192 bits and 256 bits. The security of the AES algorithm is based on the computational complexity of trying
all possible keys.<br> A larger key size increases the number of possible keys and makes exhaustive search attacks computationally infeasible.
Therefore, a 256-bit key provides a higher level of security than a 128-bit or 192-bit key.</p>
<p>Currently, a minimum key size of <strong>128 bits</strong> is recommended for AES.</p>
<h4>Elliptic Curve Cryptography (ECC)</h4>
<p>Elliptic curve cryptography is also used in various algorithms, such as ECDSA, ECDH, or ECMQV. The length of keys generated with elliptic curve
algorithms are mentioned directly in their names. For example, <code>secp256k1</code> generates a 256-bits long private key.</p>
<p>Currently, a minimum key size of <strong>224 bits</strong> is recommended for EC algorithms.</p>
<h3>Going the extra mile</h3>
<h4>Pre-Quantum Cryptography</h4>
<p>Encrypted data and communications recorded today could be decrypted in the future by an attack from a quantum computer.<br> It is important to keep
in mind that NIST-approved digital signature schemes, key agreement, and key transport may need to be replaced with secure quantum-resistant (or
"post-quantum") counterpart.</p>
<p>Thus, if data is to remain secure beyond 2030, proactive measures should be taken now to ensure its safety.</p>
<p><a href="https://www.enisa.europa.eu/publications/post-quantum-cryptography-current-state-and-quantum-mitigation">Learn more here</a>.</p>
<h2>Resources</h2>
<h3>Articles &amp; blog posts</h3>
<ul>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/standard/security/vulnerabilities-cbc-mode">Microsoft, Timing vulnerabilities with CBC-mode
  symmetric decryption using padding</a> </li>
  <li> <a href="https://en.wikipedia.org/wiki/Padding_oracle_attack">Wikipedia, Padding Oracle Attack</a> </li>
  <li> <a href="https://en.wikipedia.org/wiki/Chosen-ciphertext_attack">Wikipedia, Chosen-Ciphertext Attack</a> </li>
  <li> <a href="https://en.wikipedia.org/wiki/Chosen-plaintext_attack">Wikipedia, Chosen-Plaintext Attack</a> </li>
  <li> <a href="https://en.wikipedia.org/wiki/Semantic_security">Wikipedia, Semantically Secure Cryptosystems</a> </li>
  <li> <a href="https://en.wikipedia.org/wiki/Optimal_asymmetric_encryption_padding">Wikipedia, OAEP</a> </li>
  <li> <a href="https://en.wikipedia.org/wiki/Galois/Counter_Mode">Wikipedia, Galois/Counter Mode</a> </li>
</ul>
<h3>Standards</h3>
<ul>
  <li> <a href="https://owasp.org/Top10/A02_2021-Cryptographic_Failures/">OWASP Top 10 2021 Category A2</a> - Cryptographic Failures </li>
  <li> <a href="https://www.owasp.org/www-project-top-ten/2017/A3_2017-Sensitive_Data_Exposure">OWASP Top 10 2017 Category A3</a> - Sensitive Data
  Exposure </li>
  <li> <a href="https://owasp.org/www-project-top-ten/2017/A6_2017-Security_Misconfiguration">OWASP Top 10 2017 Category A6</a> - Security
  Misconfiguration </li>
  <li> <a href="https://mobile-security.gitbook.io/masvs/security-requirements/0x08-v3-cryptography_verification_requirements">Mobile AppSec
  Verification Standard</a> - Cryptography Requirements </li>
  <li> <a href="https://owasp.org/www-project-mobile-top-10/2016-risks/m5-insufficient-cryptography">OWASP Mobile Top 10 2016 Category M5</a> -
  Insufficient Cryptography </li>
  <li> <a href="https://nvlpubs.nist.gov/nistpubs/SpecialPublications/NIST.SP.800-131Ar1.pdf">NIST 800-131A</a> - Recommendation for Transitioning the
  Use of Cryptographic Algorithms and Key Lengths* <a href="https://cwe.mitre.org/data/definitions/326">MITRE, CWE-326</a> - Inadequate Encryption
  Strength </li>
  <li> <a href="https://cwe.mitre.org/data/definitions/327">MITRE, CWE-327</a> - Use of a Broken or Risky Cryptographic Algorithm </li>
  <li> <a href="https://wiki.sei.cmu.edu/confluence/x/hDdGBQ">CERT, MSC61-J.</a> - Do not use insecure or weak cryptographic algorithms </li>
</ul>