Duplicate statements in conditional expressions create unnecessary code redundancy.
Consider merging conditions that lead to the same body of code, to simplify the structure and improve code readability.

# What Does This Check Look For?

This check identifies cases in conditional expressions like `if`, `else if`, `else`, or `switch/case` where the statements in different branches are identical.

# Why is This a Problem?

Having identical statements in different branches of conditional constructs can be problematic for several reasons:
1. **Maintenance:** If the duplicated code needs to be changed, it must be located and updated in multiple places, which is time-consuming and error-prone.
2. **Readability:** Redundant code can make the codebase more difficult to navigate and understand, obscuring the logic and flow of the program.
3. **Efficiency:** Merging conditions or refactoring can lead to more concise and efficient code, reducing the overall size of the codebase and potentially improving performance.


# How Can I Resolve This?

- **Merge Conditions:** If multiple branches perform the same action, consider merging their conditions using logical operators (e.g., `||` for OR).
- **Extract Method:** For complex or repeated conditional logic, extracting the repeated code into a separate method can improve readability and make maintenance easier.
- **Use Early Returns:** When possible, use early returns for identical branches that end execution (e.g., with `return`, `break`, or `continue`), which can reduce nesting and clarify the conditions under which the method exits.

# Examples
## Example 1: Conditional Statements with Redundant Actions

### Non-Compliant

```js
if (condition1) {
  foo();
} else if (condition2) {
  foo();
} else {
  bar();
}
```

### Compliant

```js
if (condition1 || condition2) {
  foo();
} else {
  bar();
}
```


## Example 2: Switch Statement with Repetitive Actions

### Non-Compliant

```js
switch (x) {
	case 1:
		foo1();
		foo2();
		break;
	case 2:
		foo1();
		foo2();
		break;
	default:
		bar();
		break;
}
```

### Compliant

```js
switch (x) {
	case 1: // Fallthrough intended
	case 2:
		foo1();
		foo2();
		break;
	default:
		bar();
		break;
}
```

## Example3: Complex Conditions

### Non-Compliant

```js
if ((complex && long) || (condition && number(1)) {
  foo();
} else if ((complex && long) || (condition && number(2)) {
  foo();
}
```

### Compliant

```js
// easier to understand:
let isComplexAndLong = complex && long;
let complexCondition1Holds = isComplexAndLong || (condition && number(1))
let complexCondition2Holds = isComplexAndLong || (condition && number(2))
if (complexCondition1Holds || complexCondition2Holds) {
	foo();
}
```

