Using non-ASCII characters in identifiers can lead to compatibility issues
and make the code difficult to read and maintain.
Instead, it is recommended to use ASCII characters and follow naming conventions that are clear and meaningful.

# What Does This Check Look For?

This check detects non-ASCII characters in identifiers such as variable names, function names, and class names.

# Why is This a Problem?

Non-ASCII characters can lead to several issues:
1. **Encoding.**
    In the past, before a standard character encoding system was established,
    different regions and countries used different character sets for non-ASCII characters such as Δ or Λ.
    This caused inconsistencies in the encoding of characters across different regions,
    i.e., the identifier for a particular character in one region could represent a different character in another region.
2. **Tool Support.**
    While most compilers recognize non-ASCII characters, there is a possibility that your compiler might not support them.
3. **Internationalization.**
    Exclusively using ASCII characters and english vocabulary in the codebase enables international users
    to understand and work on the code. 
    Using non-ASCII characters might confuse developers trying to understand your code.
4. **Unicode interpretation.**
    Unicode characters are interpreted differently depending on the programming language:
    In Swift, C++, and C#, spacing characters are interpreted as token separators,
    whereas Python considers them as errors.
    Additionally, Swift and C++ allow the use of emojis as identifiers, while Python and C# do not.
5. **Lookalikes.**
    Unicode can lead to text or code that looks the same but uses different characters.
    Ligatures are combinations of two letters, which can be represented in unicode as a single character.
    This can lead to misleading code, where multiple variables look the same,
    but one of them uses ligatures and the other one does not.
    Another example is roman numbers which could be mistaken for letters like I or L.

# How Can I Resolve This?

Instead of using non-ASCII characters for symbols and certain letters in identifiers, you could spell them out. 
An alternative for umlauts is spelling them as their base-vowels, e.g., ae instead of &auml; or ue instead of &uuml;.

# Examples
## Non-Compliant
This example uses Java, but the principle generalizes to other languages.

```java
public static int[] price(){
		int maisstärke = 10;
		int in£ = 2;
		int in€ = 1.5;
		
		int priceUK = maisstärke * in£;
		int priceEuropean = maisstärke * in€;
		return new int[] {priceUK, priceEuropean};
}
```

## Compliant

```java
public static int[] price(){
		int maisstaerke = 10;
		int inPound = 2;
		int inEuro = 1.5;

		int priceUK = maisstaerke * inPound;
		int priceEuropean = maisstaerke * inEuro;
		return new int[] {priceUK, priceEuropean};
}
```

