<h2>Why is this an issue?</h2>
<p>An <code>IDisposable</code> object should be disposed (there are some rare exceptions where not disposing is fine, most notably <code>Task</code>).
If a class has an <code>IDisposable</code> field, there can be two situations:</p>
<ul>
  <li> The class observes a field that is under the responsibility of another class. </li>
  <li> The class owns the field, and is therefore responsible for calling <code>Dispose</code> on it. </li>
</ul>
<p>In the second case, the safest way for the class to ensure <code>Dispose</code> is called is to call it in its own <code>Dispose</code> function,
and therefore to be itself <code>IDisposable</code>. A class is considered to own an <code>IDisposable</code> field resource if it created the object
referenced by the field.</p>
<h3>Noncompliant code example</h3>
<pre data-diff-id="1" data-diff-type="noncompliant">
public class ResourceHolder   // Noncompliant; doesn't implement IDisposable
{
    private FileStream fs;  // This member is never Disposed
    public void OpenResource(string path)
    {
        this.fs = new FileStream(path, FileMode.Open); // I create the FileStream, I'm owning it
    }
    public void CloseResource()
    {
        this.fs.Close();
    }
}
</pre>
<h3>Compliant solution</h3>
<pre data-diff-id="1" data-diff-type="compliant">
public class ResourceHolder : IDisposable
{
    private FileStream fs;
    public void OpenResource(string path)
    {
        this.fs = new FileStream(path, FileMode.Open); // I create the FileStream, I'm owning it
    }
    public void CloseResource()
    {
        this.fs.Close();
    }
    public void Dispose()
    {
        this.fs.Dispose();
    }
}
</pre>
<h2>Resources</h2>
<ul>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/459">CWE-459 - Incomplete Cleanup</a> </li>
</ul>

