<p>This rule raises an issue when a subclass of a class that overrides <code>Object.equals</code> introduces new fields but does not also override the
<code>Object.equals</code> method.</p>
<h2>Why is this an issue?</h2>
<p>When a class overrides <code>Object.equals</code>, this indicates that the class not just considers object identity as equal (the default
implementation of <code>Object.equals</code>) but implements another logic for what is considered equal in the context of this class. Usually (but not
necessarily), the semantics of <code>equals</code> in this case is that two objects are equal when their state is equal field by field.</p>
<p>Because of this, adding new fields to a subclass of a class that overrides <code>Object.equals</code> but not updating the implementation of
<code>equals</code> in the subclass is most likely an error.</p>
<h2>How to fix it</h2>
<p>Consider the following example:</p>

```java
class Foo {

  final int a;

  @Override
  public boolean equals(Object other) {
    if (other == null) return false;
    if (getClass() != other.getClass()) return false;
    return a == ((Foo) other).a;
  }
}
```

```java
class Bar extends Foo { // Noncompliant, `equals` ignores the value of `b`
  final int b;
}
```
<p>Override the <code>equals</code> method in the subclass to incorporate the new fields into the comparison:</p>

```java
class Bar extends Foo { // Compliant, `equals` now also considers `b`

  final int b;

  @Override
  public boolean equals(Object other) {
    if (!super.equals(other)) return false;
    return b == ((Bar) other).b;
  }
}
```
<p>In case the new fields should not be part of the comparison because they are, for example, auxiliary variables not contributing to the object value
(), still override the method to make the point clear that this was not just forgotten:</p>

```java
class Bar extends Foo { // Compliant, we do explicitly not want to take `b` into account

  final int b;

  @Override
  public boolean equals(Object other) {
    return super.equals(other);
  }
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> <a href="https://docs.oracle.com/javase/8/docs/api/java/lang/Object.html#equals-java.lang.Object-">Object.equals - Java SE 8 API
  Specification</a> </li>
</ul>