<h2>Why is this an issue?</h2>
<p>A test case without assertions ensures only that no exceptions are thrown. Beyond basic runnability, it ensures nothing about the behavior of the
code under test.</p>
<p>This rule raises an exception when no assertions from any of the following known frameworks are found in a test:</p>
<ul>
  <li> AssertJ </li>
  <li> Awaitility </li>
  <li> EasyMock </li>
  <li> Eclipse Vert.x </li>
  <li> Fest 1.x and 2.x </li>
  <li> Hamcrest </li>
  <li> JMock </li>
  <li> JMockit </li>
  <li> JUnit </li>
  <li> Mockito </li>
  <li> Rest-assured 2.x, 3.x and 4.x </li>
  <li> RxJava 1.x and 2.x </li>
  <li> Selenide </li>
  <li> Spring’s <code>org.springframework.test.web.servlet.ResultActions.andExpect()</code> and
  <code>org.springframework.test.web.servlet.ResultActions.andExpectAll()</code> </li>
  <li> Truth Framework </li>
  <li> WireMock </li>
</ul>
<p>Furthermore, as new or custom assertion frameworks may be used, the rule can be parametrized to define specific methods that will also be
considered as assertions. No issue will be raised when such methods are found in test cases. The parameter value should have the following format
<code>&lt;FullyQualifiedClassName&gt;#&lt;MethodName&gt;</code>, where <code>MethodName</code> can end with the wildcard character. For constructors,
the pattern should be <code>&lt;FullyQualifiedClassName&gt;#&lt;init&gt;</code>.</p>
<p>Example:&nbsp;<code>com.company.CompareToTester#compare*,com.company.CustomAssert#customAssertMethod,com.company.CheckVerifier#&lt;init&gt;</code>.</p>
<h3>Noncompliant code example</h3>

```java
@Test
public void testDoSomething() {  // Noncompliant
  MyClass myClass = new MyClass();
  myClass.doSomething();
}
```
<h3>Compliant solution</h3>
<p>Example when <code>com.company.CompareToTester#compare*</code> is used as parameter to the rule.</p>

```java
import com.company.CompareToTester;

@Test
public void testDoSomething() {
  MyClass myClass = new MyClass();
  assertNull(myClass.doSomething());  // JUnit assertion
  assertThat(myClass.doSomething()).isNull();  // Fest assertion
}

@Test
public void testDoSomethingElse() {
  MyClass myClass = new MyClass();
  new CompareToTester().compareWith(myClass);  // Compliant - custom assertion method defined as rule parameter
  CompareToTester.compareStatic(myClass);  // Compliant
}
```