<h2>Why is this an issue?</h2>
<p><code>Optional</code> acts as a container object that may or may not contain a non-null value. It is introduced in Java 8 to help avoid
<code>NullPointerException</code>. It provides methods to check if a value is present and retrieve the value if it is present.</p>
<p><code>Optional</code> is used instead of <code>null</code> values to make the code more readable and avoid potential errors.</p>
<p>It is a bad practice to use <code>null</code> with <code>Optional</code> because it is unclear whether a value is present or not, leading to
confusion and potential <code>NullPointerException</code> errors.</p>
<h2>How to fix it</h2>
<p>There are a few ways to fix this issue:</p>
<ul>
  <li> Avoid returning <code>null</code> from a method whose return type is <code>Optional</code>. </li>
  <li> Remove the null-check of an <code>Optional</code> and use <code>Optional</code> methods instead, like <code>isPresent()</code> or
  <code>ifPresent()</code>. </li>
</ul>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```java
public void doSomething () {
  Optional<String> optional = getOptional();
  if (optional != null) {  // Noncompliant
    // do something with optional...
  }
  Optional<String> text = null; // Noncompliant, a variable whose type is Optional should never itself be null
  // ...
}

@Nullable // Noncompliant
public Optional<String> getOptional() {
  // ...
  return null;  // Noncompliant
}
```
<h4>Compliant solution</h4>

```java
public void doSomething () {
  Optional<String> optional = getOptional();
  optional.ifPresent(
    // do something with optional...
  );
  Optional<String> text = Optional.empty();
  // ...
}

public Optional<String> getOptional() {
  // ...
  return Optional.empty();
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> <a href="https://docs.oracle.com/en/java/javase/20/docs/api/java.base/java/util/Optional.html">Oracle SDK 20 - Optional</a> </li>
</ul>
<h3>Articles &amp; blog posts</h3>
<ul>
  <li> <a href="https://www.baeldung.com/java-optional">Java Optional Guide</a> </li>
</ul>