<h2>Why is this an issue?</h2>
<p>When iterating over an <code>Iterable</code> with a <code>for</code> loop, the iteration variable could have the same type as the type returned by
the iterator (the item type of the <code>Iterable</code>). This rule reports when a supertype of the item type is used for the variable instead, but
the variable is then explicitly downcast in the loop body.</p>
<p>Using explicit type casts instead of leveraging the language’s type system is a bad practice. It disables static type checking by the compiler for
the cast expressions, but potential errors will throw a <code>ClassCastException</code> during runtime instead.</p>
<h2>How to fix it</h2>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>
<p>When declaring the iteration variable, use the item type for it instead of a supertype. Remove the explicit downcasts in the loop body.</p>

```java
for (Object item : getPersons()) { // Noncompliant, iteration element is implicitly upcast here
  Person person = (Person) item; // Noncompliant, item is explicitly downcast here
  person.getAddress();
}
```
<h4>Compliant solution</h4>

```java
for (Person person : getPersons()) { // Compliant
  person.getAddress();
}
```
<h4>Noncompliant code example</h4>
<p>Alternatively, use the <code>var</code> keyword to automatically infer the variable type (since Java 10).</p>

```java
for (Object item : getPersons()) { // Noncompliant, iteration element is implicitly upcast here
  Person person = (Person) item; // Noncompliant, item is explicitly downcast here
  person.getAddress();
}
```
<h4>Compliant solution</h4>

```java
for (var person : getPersons()) { // Compliant
  person.getAddress();
}
```
<h4>Compliant solution</h4>
<p>The implicit upcast in the loop header is not reported when there is no downcast in the loop body.</p>

```java
for (Object item : getPersons()) { // Compliant
  System.out.println(item);
}
```