<p>This vulnerability allows the usage of external entities in XML.</p>
<h2>Why is this an issue?</h2>
<p>External Entity Processing allows for XML parsing with the involvement of external entities. However, when this functionality is enabled without
proper precautions, it can lead to a vulnerability known as XML External Entity (XXE) attack.</p>
<h3>What is the potential impact?</h3>
<h4>Exposing sensitive data</h4>
<p>One significant danger of XXE vulnerabilities is the potential for sensitive data exposure. By crafting malicious XML payloads, attackers can
reference external entities that contain sensitive information, such as system files, database credentials, or configuration files. When these
entities are processed during XML parsing, the attacker can extract the contents and gain unauthorized access to sensitive data. This poses a severe
threat to the confidentiality of critical information.</p>
<h4>Exhausting system resources</h4>
<p>Another consequence of XXE vulnerabilities is the potential for denial-of-service attacks. By exploiting the ability to include external entities,
attackers can construct XML payloads that cause resource exhaustion. This can overwhelm the system’s memory, CPU, or other critical resources, leading
to system unresponsiveness or crashes. A successful DoS attack can disrupt the availability of services and negatively impact the user experience.</p>
<h4>Forging requests</h4>
<p>XXE vulnerabilities can also enable Server-Side Request Forgery (SSRF) attacks. By leveraging the ability to include external entities, an attacker
can make the vulnerable application send arbitrary requests to other internal or external systems. This can result in unintended actions, such as
retrieving data from internal resources, scanning internal networks, or attacking other systems. SSRF attacks can lead to severe consequences,
including unauthorized data access, system compromise, or even further exploitation within the network infrastructure.</p>
<h2>How to fix it in libxmljs</h2>
<h3>Code examples</h3>
<p>The following code contains examples of XML parsers that have external entity processing enabled. As a result, the parsers are vulnerable to XXE
attacks if an attacker can control the XML file that is processed.</p>
<h4>Noncompliant code example</h4>

```js
var libxmljs = require('libxmljs');
var fs = require('fs');

var xml = fs.readFileSync('xxe.xml', 'utf8');
libxmljs.parseXmlString(xml, {
    noblanks: true,
    noent: true, // Noncompliant
    nocdata: true
});
```
<h4>Compliant solution</h4>
<p><code>parseXmlString</code> is safe by default.</p>

```js
var libxmljs = require('libxmljs');
var fs = require('fs');

var xml = fs.readFileSync('xxe.xml', 'utf8');
libxmljs.parseXmlString(xml);
```
<h3>How does this work?</h3>
<h4>Disable external entities</h4>
<p>The most effective approach to prevent XXE vulnerabilities is to disable external entity processing entirely, unless it is explicitly required for
specific use cases. By default, XML parsers should be configured to reject the processing of external entities. This can be achieved by setting the
appropriate properties or options in your XML parser library or framework.</p>
<p>If external entity processing is necessary for certain scenarios, adopt a whitelisting approach to restrict the entities that can be resolved
during XML parsing. Create a list of trusted external entities and disallow all others. This approach ensures that only known and safe entities are
processed.<br> You should rely on features provided by your XML parser to restrict the external entities.</p>
<h2>Resources</h2>
<h3>Standards</h3>
<ul>
  <li> OWASP - <a href="https://owasp.org/Top10/A05_2021-Security_Misconfiguration/">Top 10 2021 Category A5 - Security Misconfiguration</a> </li>
  <li> OWASP - <a href="https://owasp.org/www-project-top-ten/2017/A4_2017-XML_External_Entities_(XXE)">Top 10 2017 Category A4 - XML External
  Entities (XXE)</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/611">CWE-611 - Information Exposure Through XML External Entity Reference</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/827">CWE-827 - Improper Control of Document Type Definition</a> </li>
  <li> STIG Viewer - <a href="https://stigviewer.com/stig/application_security_and_development/2023-06-08/finding/V-222608">Application Security and
  Development: V-222608</a> - The application must not be vulnerable to XML-oriented attacks. </li>
</ul>