<p>This vulnerability exposes encrypted data to a number of attacks whose goal is to recover the plaintext.</p>
<h2>Why is this an issue?</h2>
<p>Encryption algorithms are essential for protecting sensitive information and ensuring secure communications in a variety of domains. They are used
for several important reasons:</p>
<ul>
  <li> Confidentiality, privacy, and intellectual property protection </li>
  <li> Security during transmission or on storage devices </li>
  <li> Data integrity, general trust, and authentication </li>
</ul>
<p>When selecting encryption algorithms, tools, or combinations, you should also consider two things:</p>
<ol>
  <li> No encryption is unbreakable. </li>
  <li> The strength of an encryption algorithm is usually measured by the effort required to crack it within a reasonable time frame. </li>
</ol>
<p>For these reasons, as soon as cryptography is included in a project, it is important to choose encryption algorithms that are considered strong and
secure by the cryptography community.</p>
<p>For AES, the weakest modes are CBC (Cipher Block Chaining) and ECB</p>
<p>(Electronic Codebook), as they are either vulnerable to padding oracles or do not provide authentication mechanisms.</p>
<p>And for RSA, the weakest algorithms are either using it without padding or using the PKCS1v1.5 padding scheme.</p>
<h3>What is the potential impact?</h3>
<p>The cleartext of an encrypted message might be recoverable. Additionally, it might be possible to modify the cleartext of an encrypted message.</p>
<p>Below are some real-world scenarios that illustrate possible impacts of an attacker exploiting the vulnerability.</p>
<h4>Theft of sensitive data</h4>
<p>The encrypted message might contain data that is considered sensitive and should not be known to third parties.</p>
<p>By using a weak algorithm the likelihood that an attacker might be able to recover the cleartext drastically increases.</p>
<h4>Additional attack surface</h4>
<p>By modifying the cleartext of the encrypted message it might be possible for an attacker to trigger other vulnerabilities in the code. Encrypted
values are often considered trusted, since under normal circumstances it would not be possible for a third party to modify them.</p>
<h2>How to fix it in Java Cryptographic Extension</h2>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>
<p>Example with a symmetric cipher, AES:</p>

```kotlin
import javax.crypto.Cipher
import javax.crypto.NoSuchPaddingException
import java.security.NoSuchAlgorithmException

fun main(args: Array<String>) {
    try {
        val aes = Cipher.getInstance("AES/CBC/PKCS5Padding"); // Noncompliant
    } catch (e: NoSuchAlgorithmException) {
        // ...
    } catch (e: NoSuchPaddingException) {
        // ...
    }
}
```
<p>Example with an asymmetric cipher, RSA:</p>

```kotlin
import javax.crypto.Cipher
import javax.crypto.NoSuchPaddingException
import java.security.NoSuchAlgorithmException

fun main(args: Array<String>) {
    try {
        val rsa = Cipher.getInstance("RSA/None/NoPadding"); // Noncompliant
    } catch (e: NoSuchAlgorithmException) {
        // ...
    } catch (e: NoSuchPaddingException) {
        // ...
    }
}
```
<h4>Compliant solution</h4>
<p>For the AES symmetric cipher, use the GCM mode:</p>

```kotlin
import javax.crypto.Cipher
import javax.crypto.NoSuchPaddingException
import java.security.NoSuchAlgorithmException

fun main(args: Array<String>) {
    try {
        val aes = Cipher.getInstance("AES/GCM/NoPadding");
    } catch (e: NoSuchAlgorithmException) {
        // ...
    } catch (e: NoSuchPaddingException) {
        // ...
    }
}
```
<p>For the RSA asymmetric cipher, use the Optimal Asymmetric Encryption Padding (OAEP):</p>

```kotlin
import javax.crypto.Cipher
import javax.crypto.NoSuchPaddingException
import java.security.NoSuchAlgorithmException

fun main(args: Array<String>) {
    try {
        val rsa = Cipher.getInstance("RSA/ECB/OAEPWITHSHA-256ANDMGF1PADDING");
    } catch (e: NoSuchAlgorithmException) {
        // ...
    } catch (e: NoSuchPaddingException) {
        // ...
    }
}
```
<h3>How does this work?</h3>
<p>As a rule of thumb, use the cryptographic algorithms and mechanisms that are considered strong by the cryptographic community.</p>
<p>Appropriate choices are currently the following.</p>
<h4>For AES: Use Galois/Counter mode (GCM)</h4>
<p>GCM mode combines encryption with authentication and integrity checks using a cryptographic hash function and provides both confidentiality and
authenticity of data.</p>
<p>Other similar modes are:</p>
<ul>
  <li> CCM: <code>Counter with CBC-MAC</code> </li>
  <li> CWC: <code>Cipher Block Chaining with Message Authentication Code</code> </li>
  <li> EAX: <code>Encrypt-and-Authenticate</code> </li>
  <li> IAPM: <code>Integer Authenticated Parallelizable Mode</code> </li>
  <li> OCB: <code>Offset Codebook Mode</code> </li>
</ul>
<p>It is also possible to use AES-CBC with HMAC for integrity checks. However, it</p>
<p>is considered more straightforward to use AES-GCM directly instead.</p>
<h4>For RSA: use the OAEP scheme</h4>
<p>The Optimal Asymmetric Encryption Padding scheme (OAEP) adds randomness and a secure hash function that strengthens the regular inner workings of
RSA.</p>
<h2>Resources</h2>
<h3>Articles &amp; blog posts</h3>
<ul>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/standard/security/vulnerabilities-cbc-mode">Microsoft, Timing vulnerabilities with CBC-mode
  symmetric decryption using padding</a> </li>
  <li> <a href="https://en.wikipedia.org/wiki/Padding_oracle_attack">Wikipedia, Padding Oracle Attack</a> </li>
  <li> <a href="https://en.wikipedia.org/wiki/Chosen-ciphertext_attack">Wikipedia, Chosen-Ciphertext Attack</a> </li>
  <li> <a href="https://en.wikipedia.org/wiki/Chosen-plaintext_attack">Wikipedia, Chosen-Plaintext Attack</a> </li>
  <li> <a href="https://en.wikipedia.org/wiki/Semantic_security">Wikipedia, Semantically Secure Cryptosystems</a> </li>
  <li> <a href="https://en.wikipedia.org/wiki/Optimal_asymmetric_encryption_padding">Wikipedia, OAEP</a> </li>
  <li> <a href="https://en.wikipedia.org/wiki/Galois/Counter_Mode">Wikipedia, Galois/Counter Mode</a> </li>
</ul>
<h3>Standards</h3>
<ul>
  <li> <a href="https://owasp.org/Top10/A02_2021-Cryptographic_Failures/">OWASP Top 10 2021 Category A2</a> - Cryptographic Failures </li>
  <li> <a href="https://owasp.org/www-project-top-ten/2017/A6_2017-Security_Misconfiguration">OWASP Top 10 2017 Category A6</a> - Security
  Misconfiguration </li>
  <li> <a href="https://cwe.mitre.org/data/definitions/327">MITRE, CWE-327</a> - Use of a Broken or Risky Cryptographic Algorithm </li>
  <li> <a href="https://mobile-security.gitbook.io/masvs/security-requirements/0x08-v3-cryptography_verification_requirements">Mobile AppSec
  Verification Standard</a> - Cryptography Requirements </li>
  <li> <a href="https://owasp.org/www-project-mobile-top-10/2016-risks/m5-insufficient-cryptography">OWASP Mobile Top 10 2016 Category M5</a> -
  Insufficient Cryptography </li>
  <li> <a href="https://cwe.mitre.org/data/definitions/327">MITRE, CWE-327</a> - Use of a Broken or Risky Cryptographic Algorithm </li>
  <li> <a href="https://wiki.sei.cmu.edu/confluence/x/hDdGBQ">CERT, MSC61-J.</a> - Do not use insecure or weak cryptographic algorithms </li>
</ul>