<h2>Why is this an issue?</h2>
<p>The primary purpose of classes that implement indexed access operators is that of element access, like this is the case for arrays, lists, maps, or
sets. When a class implements indexed access operators, they should be used as operators instead of calling them as functions, because this is the
intention of the designer of the API.</p>
<h3>What is the potential impact?</h3>
<h4>Readability and Understanding</h4>
<p>This change makes it easier to understand what a function does, because the semantics of indexed access operators is evident to the reader, while
for a function call, the reader would need to know what the called function does.</p>
<h4>Java interop</h4>
<p>The issue is also raised when Java API is used as a Kotlin compiler treats all <code>get(.)</code> methods as possible index access operators. Use
<code>CompletableFuture[timeout, unit]</code> instead of <code>CompletableFuture.get(timeout, unit)</code> when accessing Java’s
<code>CompletableFuture</code>.</p>
<h2>How to fix it</h2>
<p>Replace <code>instance.get(index)</code> with <code>instance[index]</code>.</p>
<p>Replace <code>instance.set(index, value)</code> with <code>instance[index] = value</code>.</p>
<p>This also works with multi-index access operators.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```kotlin
interface Grid {
    operator fun get(row: Int, column: Int): Int
    operator fun set(row: Int, column: Int, value: Int)
}
```

```kotlin
fun indexedAccess(list: MutableList<Int>, map: MutableMap<String, Int>, grid: Grid) {
    list.get(1) // Noncompliant
    list.set(1, 42) // Noncompliant
    map.get("b") // Noncompliant
    map.set("b", 42) // Noncompliant
    grid.get(1, 2) // Noncompliant
    grid.set(1, 2, 42) // Noncompliant
}
```
<h4>Compliant solution</h4>

```kotlin
fun indexedAccess(list: MutableList<Int>, map: MutableMap<String, Int>, grid: Grid) {
    list[1] // Compliant
    list[1] = 42 // Compliant
    map["b"] // Compliant
    map["b"] = 42 // Compliant
    grid[1, 2] // Compliant
    grid[1, 2] = 42 // Compliant
    list.getOrNull(2) // Compliant, because function is not an operator
    list.getOrElse(3) {42} // Compliant, because function is not an operator
    map.getValue("a") // Compliant, because function is not an operator
    map.getOrElse("c") {42} // Compliant, because function is not an operator
}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> <a href="https://kotlinlang.org/docs/operator-overloading.html#indexed-access-operator">Kotlin Docs, Indexed access operator</a> </li>
</ul>