Dereferencing a potentially null pointer or null reference can cause runtime exceptions and unexpected behavior.
Consider performing null checks before dereferencing pointers or using safer dereferencing methods.

# What Does This Check Look For?
This check looks for null dereferencing problems and identifies potential instances where a program may attempt to access or manipulate an object or variable that has not been properly initialized or has been set to `null` (in Java and C#) or `NULL`, `0`, or `nullptr` (in C and C++).

Some notes related to C/C++ code: first, note that it is sometimes common to dereference a pointer after calling `malloc`.
However, since `malloc` can fail, Teamscale will create a finding if a pointer is dereferenced after `malloc` without a prior null check. 
Second, Teamscale will identify the termination of a possible control flow when control-flow termination keywords are used (such as `return` or `break`) or when the following functions are called `exit()`, `abort()`, `terminate()`, `quick_exit()`, and `_Exit()`. 

# Why is This a Problem?
Dereferencing a null pointer is a common cause of runtime errors, such as segmentation faults, program crashes, or exception throw (depending on the language). 
This issue affects the reliability and stability of an application, and can also pose security risks if not handled properly.
Moreover, null pointer dereference might occur sporadically, only in certain conditions, making them hard to trace and fix. 
Proper handling of null pointers is crucial for ensuring that the application behaves as expected under all conditions.

# How Can I Resolve This?
There are different ways to ensure a pointer is checked for nullability before it is dereferenced, depending on the programming environment.
The most simple solution might be a null check with an `if` statement before pointer dereference.
If the application logic allows, consider changing the code to eliminate the need for a pointer to ever be null at the point of dereference.  

Alternatively, use assertions or exception handling with logging to catch unexpected null values defensively.

Also, a common practice is to use custom assertTrue-like functions to check for pointer nullability before dereference.
If using a preprocessed language (e.g. C/C+), and these assertTrue-like functions are defined as macros, make sure those macros are visible to Teamscale.
For Java or C#, see the check option to pass a list of such custom methods.

Finally, it is also possible to set another check option to list methods that are allowed to be called on null and thus should not cause null-pointer dereference findings.

# Examples

## Non-Compliant
```c
// In C:
Foo *ptr = (Foo*) malloc(sizeof(*ptr));
int x = ptr->bar // non-compliant: possible null pointer dereference if malloc fails
if (ptr) {
  // do something
}
exit(1);
```

```c
// In C:
int print (struct Foo *fizz) {
   printf("Printing: %c", fizz->buzz); // non-compliant: possible null pointer dereference since this is before null check
   if (fizz == NULL) 
      return 0;
   
   return 1;
}
```

```c#
// In C#:
public int WriteMessageLength(string? message)
{
   Console.WriteLine(message.Length); // non-compliant: will call Length if message is null

   if (message is null) {
     Console.WriteLine("Invalid msg!"); 
     return 0;
   } 

   return 1;
}
```

```java
// In Java:
public class Foo {
	public static void foo() {
		String text = null;
		boolean condition = Foo.isBar();
		while(condition) {
			text = Foo.readText();
			condition = Foo.isBar();
		}
		int length = text.length(); // Non-compliant: may call length() on a null reference, for instance, if the while does not run
		System.out.println("Length of the text is: " + length);
	}
}
```

## Compliant
```c
// In C:
Foo *ptr = (Foo*) malloc(sizeof(*ptr));
int x = 0;
if (ptr == NULL) {
   // handle exception
   return 0;
}
x = ptr->bar; // compliant: dereference after explicit null check
```

```c
// In C:
#define assertTrue(x) if ((x) == NULL) return 0

Foo *ptr = (Foo*) malloc(sizeof(*ptr));
int x = 0;
assertTrue(ptr);
x = ptr->bar; // compliant: dereference after null check in the assertTrue function defined as macro 
```

```c
// In C:
int print (struct Foo *fizz) {
   if (fizz == NULL) 
      return 0;
      
   printf("Printing: %c", fizz->buzz); // compliant: dereference after null check
   return 1;
}
```

```c#
// In C#:
public int WriteMessageLength(string? message)
{
   if (message is null) {
     Console.WriteLine("Invalid msg!"); 
     return 0;
   } 
   
   Console.WriteLine(message.Length); // compliant: will call Length only if message is not null
   return 1;
}
```

```java
// In Java:
public class Foo {
    public static void foo() {
        String text = null;
		boolean condition = Foo.isBar();
		while(condition) {
			text = Foo.readText();
			condition = Foo.isBar();
		}
		if (text != null) {
			int length = text.length(); // compliant: guarded by the if statement
			System.out.println("Length of the text is: " + length);
		}
	}
}
```
```java
// In Java:
import java.util.Optional;

public class Foo {
    public static void foo() {
        Optional<String> textOptional = Optional.empty();
		boolean condition = Foo.isBar();
		while(condition) {
			String text = Foo.readText();
            textOptional = Optional.ofNullable(text);
            condition = Foo.isBar();
		}
        int length = textOptional.map(String::length).orElse(0); // compliant: provides a default value if text is null
        System.out.println("Length of the text is: " + length);
    }
}
```

# Where Can I Learn More?
- [SEI CERT C Coding Standard: EXP34-C. Do not dereference null pointers](https://wiki.sei.cmu.edu/confluence/display/c/EXP34-C.+Do+not+dereference+null+pointers)
- [C++ Core Guidelines: F.23 - Use a not_null\<T> to indicate that "null" is not a valid value](https://github.com/isocpp/CppCoreGuidelines/blob/master/CppCoreGuidelines.md#f23-use-a-not_nullt-to-indicate-that-null-is-not-a-valid-value)
- [ISO C++ Memory Management](https://isocpp.org/wiki/faq/freestore-mgmt)
- [Tired of Null Pointer Exceptions? Consider Using Java SE 8's "Optional"!](https://www.oracle.com/technical-resources/articles/java/java8-optional.html)