Duplication in conditional constructs reduces readability and makes maintenance harder.
It is better to merge conditions with the same body or refactor for more clarity.
This check discourages identical statements in the body of `if`, `else`, `else-if`, or `switch/case` constructs in Swift.

# What Does This Check Look For?

This check reports a finding when identical actions or statements are written in multiple branches of `if`,
`else`, `else-if`, or `switch/case` constructs.

# Why is This a Problem?

Identical code in multiple conditional branches not only violates the DRY (Do not Repeat Yourself) principle,
it also usually results in harder-to-read and harder-to-maintain code.
If a change is needed, you will have to update it in several places, increasing the chance of making an error.

# How Can I Resolve This?

You can merge the conditions that lead to the identical action.
For `if` and `else-if` constructs, you can use logical operators (`||`, `&&`) to combine conditions.
For `switch/case` constructs, you can use the `fallthrough` keyword to cascade cases with the same body.

If your conditions are complex and side-effect-free, consider extracting the result into a boolean variable with a descriptive name.
This makes the code more legible.

If all identical branches contain only a `return`, `continue`, `break`, or similar, you can often convert them to early returns.
This is more concise and explains that processing should not continue unless the specified condition is met.

# Examples
## Non-Compliant

```swift
if condition1 {
  foo()
} else if condition2 { 
  foo()
} else { 
  bar()
}

switch x {
case 1:
  foo1()
  foo2()
  break
case 2:
  foo1()
  foo2()
  break
default:
  bar()
  break
}
```

## Compliant

```swift
if condition1 || condition2 { 
  foo() 
} else { 
  bar() 
}

switch x {
case 1:
  fallthrough // will execute the next case
case 2:
  foo1()
  foo2()
  break
default:
  bar()
  break
}
```

# Where Can I Learn More?

- (Guideline) Swift.org - [The Swift Programming Language (6.0 beta): Control Flow](https://docs.swift.org/swift-book/documentation/the-swift-programming-language/controlflow/)

Please note: In Swift, unlike in C and many other languages, `switch/case` does not "fallthrough" by default.
Swift requires explicit `fallthrough` keyword to continue to the next case. If a case body is just one line long,
duplicating the line instead of using `fallthrough` can often be clearer and easier to understand.