Using Java’s `assert` statement for validation is discouraged in production code 
because assertions can be disabled at runtime.

# What Does This Check Look For?

This check detects usage of the `assert` keyword in Java code. 
Assertions are a language feature that allows developers to document 
and automatically validate their assumptions about the program state during development and testing.

# Why is This a Problem?

Statements using `assert` are disabled by default at runtime
unless assertions are explicitly enabled using JVM options (e.g., `-ea`).

Therefore, assertions are not guaranteed to run in all environments and could be silently skipped, 
making them an unreliable mechanism for implementing logic that must always hold.
This includes validating method arguments, enforcing invariants, or preventing invalid states. 
`assert` statements can create false confidence about the integrity of the application, 
as production environments often run with assertions turned off.

This inconsistent handling can introduce bugs and security vulnerabilities.

# How Can I Resolve This?

Replace `assert` statements with explicit conditional tests that throw an appropriate exception, 
such as `IllegalArgumentException` or `IllegalStateException`. 
Alternatively, use a validation utility method or framework 
(e.g., `java.util.Objects.requireNonNull` or `javax.validation`) to enforce conditions.

Using exceptions or validation methods ensures that the validation logic is always executed 
and violations are caught, handled and reported consistently, regardless of JVM configuration.
It also results in clearer, more maintainable code and encourages proper error handling strategies.

# Examples

## Non-Compliant

```java
public class OrderService {
    public void process(Order order) {
        assert order != null; // non-compliant: assertion may be skipped at runtime
        // business logic here
    }
}
```

## Compliant

```java
public class OrderService {
    public void process(Order order) {
        if (order == null) {
            throw new IllegalArgumentException("Order must not be null");
        }
        // business logic here
    }
}
```

Or, using utility validation:

```java
import java.util.Objects;

public class OrderService {
    public void process(Order order) {
        Objects.requireNonNull(order, "Order must not be null");
        // business logic here
    }
}
```

# Where Can I Learn More?

- [Oracle Java Language Specification – Assertion Statements](https://docs.oracle.com/javase/specs/jls/se20/html/jls-14.html#jls-14.10)

- [Baeldung – Guide to Java assert](https://www.baeldung.com/java-assert)
