Method definitions in header files can cause linkage error, expose implementation details, and hinder encapsulation.
Use `inline` functions when appropriate or move the function definition to a source file.

# Why is this a problem?

Function definitions in header files should be avoided for the following reasons:
1. **Linkage errors**: If the function is not marked as `static` and appears in more than one translation unit, it can cause linkage errors.
2. **Violation of information hiding principle**: Function definitions should not be part of the public interface in header files as this can expose implementation details and hinder encapsulation.
3. **Frequent recompiling**: Whenever a method definition in the header file is changed, all files that include it must be recompiled, leading to unnecessary compile-time overhead.

# How Can I Resolve This?

- Move the function definition to a source file.
- Mark the function as `inline` if the function needs to be available in multiple translation units.

# Examples
## Non-Compliant

```CPP
// MathUtils.h
int multiply(int a, int b) {
    return a * b;
}
```

## Compliant

```CPP
// MathUtils.h
inline int multiply(int a, int b) {
    return a * b;
}
```

# Where Can I Learn More?

* [Guideline: C++ Core Guidelines: SF.2: A header file must not contain object definitions or non-inline function definitions](https://isocpp.github.io/CppCoreGuidelines/CppCoreGuidelines#Rs-inline)
* Guideline: MISRA C:2012: 8.10 An inline function shall be declared with the static storage class