Classes should not have public attributes as this violates encapsulation.
Instead, access should be provided using public methods (getters and setters).

# What Does This Check Look For?

This check looks for public fields (attributes) in classes that should be encapsulated with private visibility and accessed through public methods instead.

# Why is This a Problem?

Public attributes violate the principle of encapsulation, which is a fundamental concept in object-oriented programming.
When fields are public, they can be directly accessed and modified by any code that has access to the class instance.
This creates several issues:
- **Loss of control**: The class cannot validate or control how its data is modified
- **Tight coupling**: Other code becomes directly dependent on the internal structure of the class
- **Maintenance difficulties**: Changing the field name or type becomes a breaking change for all code using it
- **No validation**: There's no way to ensure data integrity when values are set
- **Debugging challenges**: It becomes harder to track where and when field values are modified

# How Can I Resolve This?

Make the field private and provide public getter and setter methods to control access:
- Use `private` visibility for the field
- Create a public getter method to read the value
- Create a public setter method to modify the value (if modification should be allowed)
- Add validation logic in setter methods if needed

# Examples

## Non-Compliant

```java
public class Person {
    public String name;
    public int age;
    public String email;
}
```

The above allows direct access to all fields, making it impossible to validate data or track changes.

## Compliant

```java
public class Person {
    private String name;
    private int age;
    private String email;
    
    public String getName() {
        return name;
    }
    
    public void setName(String name) {
        if (name == null || name.trim().isEmpty()) {
            throw new IllegalArgumentException("Name cannot be null or empty");
        }
        this.name = name;
    }
    
    public int getAge() {
        return age;
    }
    
    public void setAge(int age) {
        if (age < 0 || age > 150) {
            throw new IllegalArgumentException("Age must be between 0 and 150");
        }
        this.age = age;
    }
    
    public String getEmail() {
        return email;
    }
    
    public void setEmail(String email) {
        if (email != null && !email.contains("@")) {
            throw new IllegalArgumentException("Invalid email format");
        }
        this.email = email;
    }
}
```

# Where Can I Learn More?

* [Oracle Java Documentation: Controlling Access to Members of a Class](https://docs.oracle.com/javase/tutorial/java/javaOO/accesscontrol.html)