<p>XML signatures are a method used to ensure the integrity and authenticity of XML documents. However, if XML signatures are not validated securely,
it can lead to potential vulnerabilities.</p>
<h2>Why is this an issue?</h2>
<p>Before Java 17, XML Digital Signature API does not apply restrictions on XML signature validation unless the application runs with a security
manager, which is rare.</p>
<h2>What is the potential impact</h2>
<p>By not enforcing secure validation, the XML Digital Signature API is more susceptible to attacks such as signature spoofing and injections.</p>
<h3>Increased Vulnerability to Signature Spoofing</h3>
<p>By disabling secure validation, the Java application becomes more susceptible to signature spoofing attacks. Attackers can potentially manipulate
the XML signature in a way that bypasses the validation process, allowing them to forge or tamper with the signature. This can lead to the acceptance
of invalid or maliciously modified signatures, compromising the integrity and authenticity of the XML documents.</p>
<h3>Risk of Injection Attacks</h3>
<p>Disabling secure validation can expose the application to injection attacks. Attackers can inject malicious code or entities into the XML document,
taking advantage of the weakened validation process. In some cases, it can also expose the application to denial-of-service attacks. Attackers can
exploit vulnerabilities in the validation process to cause excessive resource consumption or system crashes, leading to service unavailability or
disruption.</p>
<h2>How to fix it in Java SE</h2>
<h3>Code examples</h3>
<p>For versions of Java before 17, secure validation is disabled by default unless the application runs with a security manager, which is rare. It
should be enabled explicitly by setting the <code>org.jcp.xml.dsig.secureValidation</code> attribute to true with the
<code>javax.xml.crypto.dsig.dom.DOMValidateContext.setProperty</code> method.</p>
<p>For Java 17 and higher, secure validation is enabled by default.</p>
<h4>Noncompliant code example</h4>

```java
NodeList signatureElement = doc.getElementsByTagNameNS(XMLSignature.XMLNS, "Signature");

XMLSignatureFactory fac = XMLSignatureFactory.getInstance("DOM");
DOMValidateContext valContext = new DOMValidateContext(new KeyValueKeySelector(), signatureElement.item(0)); // Noncompliant
XMLSignature signature = fac.unmarshalXMLSignature(valContext);

boolean signatureValidity = signature.validate(valContext);
```
<h4>Compliant solution</h4>

```java
NodeList signatureElement = doc.getElementsByTagNameNS(XMLSignature.XMLNS, "Signature");

XMLSignatureFactory fac = XMLSignatureFactory.getInstance("DOM");
DOMValidateContext valContext = new DOMValidateContext(new KeyValueKeySelector(), signatureElement.item(0));
valContext.setProperty("org.jcp.xml.dsig.secureValidation", Boolean.TRUE);
XMLSignature signature = fac.unmarshalXMLSignature(valContext);

boolean signatureValidity = signature.validate(valContext);
```
<h3>How does this work?</h3>
<p>When XML Signature secure validation mode is enabled, XML Signatures are processed more securely. It enforces a number of restrictionsto to protect
from XML Documents that may contain hostile constructs that can cause denial-of-service or other types of security issues.</p>
<p>These restrictions can protect you from XML Signatures that may contain potentially hostile constructs that can cause denial-of-service or other
types of security issues.</p>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> Oracle Java Documentation - <a
  href="https://docs.oracle.com/en/java/javase/21/security/java-xml-digital-signature-api-overview-and-tutorial.html">XML Digital Signature API
  Overview and Tutorial</a> </li>
</ul>
<h3>Standards</h3>
<ul>
  <li> OWASP - <a href="https://owasp.org/Top10/A02_2021-Cryptographic_Failures/">Top 10:2021 A02:2021 - Cryptographic Failures</a> </li>
  <li> OWASP - <a href="https://owasp.org/www-project-top-ten/2017/A3_2017-Sensitive_Data_Exposure">Top 10 2017 Category A3 - Sensitive Data
  Exposure</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/347">CWE-347 - Improper Verification of Cryptographic Signature</a> </li>
</ul>