<h2>Why is this an issue?</h2>
<p>Returning <code>null</code> instead of an actual array, collection or map forces callers of the method to explicitly test for nullity, making them
more complex and less readable.</p>
<p>Moreover, in many cases, <code>null</code> is used as a synonym for empty.</p>
<h3>Noncompliant code example</h3>

```java
public static List<Result> getAllResults() {
  return null;                             // Noncompliant
}

public static Result[] getResults() {
  return null;                             // Noncompliant
}

public static Map<String, Object> getValues() {
  return null;                             // Noncompliant
}

public static void main(String[] args) {
  Result[] results = getResults();
  if (results != null) {                   // Nullity test required to prevent NPE
    for (Result result: results) {
      /* ... */
    }
  }

  List<Result> allResults = getAllResults();
  if (allResults != null) {                // Nullity test required to prevent NPE
    for (Result result: allResults) {
      /* ... */
    }
  }

  Map<String, Object> values = getValues();
  if (values != null) {                   // Nullity test required to prevent NPE
    values.forEach((k, v) -> doSomething(k, v));
  }
}
```
<h3>Compliant solution</h3>

```java
public static List<Result> getAllResults() {
  return Collections.emptyList();          // Compliant
}

public static Result[] getResults() {
  return new Result[0];                    // Compliant
}

public static Map<String, Object> getValues() {
  return Collections.emptyMap();           // Compliant
}

public static void main(String[] args) {
  for (Result result: getAllResults()) {
    /* ... */
  }

  for (Result result: getResults()) {
    /* ... */
  }

  getValues().forEach((k, v) -> doSomething(k, v));
}
```
<h2>Resources</h2>
<ul>
  <li> <a href="https://wiki.sei.cmu.edu/confluence/x/aNYxBQ">CERT, MSC19-C.</a> - For functions that return an array, prefer returning an empty array
  over a null value </li>
  <li> <a href="https://wiki.sei.cmu.edu/confluence/x/JzZGBQ">CERT, MET55-J.</a> - Return an empty array or collection instead of a null value for
  methods that return an array or collection </li>
</ul>