<h2>Why is this an issue?</h2>
<p>The <code>java.util.Collection</code> type and its subtypes provide methods to access and modify collections such as <code>Collection.remove(Object
o)</code> and <code>Collection.contains(Object o)</code>. Some of these methods accept arguments of type <code>java.lang.Object</code> and will
compare said argument with objects already in the collection.</p>
<p>If the actual type of the argument is unrelated to the type of object contained in the collection, these methods will always return
<code>false</code>, <code>null</code>, or <code>-1</code>. This behavior is most likely unintended and can be indicative of a design issue.</p>
<p>This rule raises an issue when the type of the argument provided to one of the following methods is unrelated to the type used for the collection
declaration:</p>
<ul>
  <li> <code>Collection.remove(Object o)</code> </li>
  <li> <code>Collection.removeAll(Collection&lt;?&gt;)</code> </li>
  <li> <code>Collection.contains(Object o)</code> </li>
  <li> <code>List.indexOf(Object o)</code> </li>
  <li> <code>List.lastIndexOf(Object o)</code> </li>
  <li> <code>Map.containsKey(Object key)</code> </li>
  <li> <code>Map.containsValue(Object value)</code> </li>
  <li> <code>Map.get(Object key)</code> </li>
  <li> <code>Map.getOrDefault(Object key, V defaultValue)</code> </li>
  <li> <code>Map.remove(Object key)</code> </li>
  <li> <code>Map.remove(Object key, Object value)</code> </li>
</ul>
<h2>How to fix it</h2>
<p>Ask yourself what the purpose of this method call is. Check whether the provided argument and collection are correct in this context and for the
desired purpose. Remove unnecessary calls and otherwise provide an argument of which the type is compatible with the list content’s type.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```java
void removeFromMap(Map<Integer, Object> map, String strKey) {
  map.remove(strKey); // Noncompliant, this call will remove nothing and always return 'null' because 'map' is handling only Integer keys and String cannot be cast to Integer.
}

void listContains(List<String> list, Integer integer) {
  if (list.contains(integer)) { // Noncompliant; always false as the list only contains Strings, not integers.
    // ...
  }
}
```
<h4>Compliant solution</h4>

```java
void removeFromMap(Map<Integer, Object> map, String strKey) {
  map.remove(Integer.parseInt(strKey)); // Compliant, strKey is parsed into an Integer before trying to remove it from the map.
}

void listContains(List<String> list, Integer integer) {
  if (list.contains(integer.toString())) { // Compliant, 'integer' is converted to a String before checking if the list contains it.
    // ...
  }
}
```
<h2>Resources</h2>
<ul>
  <li> <a href="https://wiki.sei.cmu.edu/confluence/x/uDdGBQ">CERT, EXP04-J.</a> - Do not pass arguments to certain Java Collections Framework methods
  that are a different type than the collection parameter type </li>
  <li> <a href="https://docs.oracle.com/en/java/javase/17/docs/api/java.base/java/util/Collection.html">Java SE 17 &amp; JDK 17</a> - Collection
  interface </li>
</ul>