<h2>Why is this an issue?</h2>
<p>Java 21 virtual threads allow the JVM to optimize the usage of OS threads, by mounting and unmounting them on an OS thread when needed, and making
them more efficient when dealing with blocking operations such as HTTP requests or I/O.</p>
<p>However, when code is executed inside a <code>synchronized</code> block or <code>synchronized</code> method, the virtual thread stays pinned to the
underlying OS thread and cannot be unmounted during a blocking operation. This will cause the OS thread to be blocked, which can impact the
scalability of the application.</p>
<p>Therefore, virtual threads should not execute code that contains <code>synchronized</code> blocks or invokes <code>synchronized</code> methods.
Platform threads should be used in these cases.</p>
<p>This rule raises an issue when a virtual thread contains <code>synchronized</code> blocks or invokes <code>synchronized</code> methods.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```java
void enqueue(){
    Thread.startVirtualThread(() -> { // Noncompliant; use a platform thread instead
            setupOperations();
            dequeLogic();
        }
    });
}
```
<h4>Compliant solution</h4>

```java
void enqueue(){
    new Thread(() -> {
        synchronized {
            setupOperations();
            dequeLogic();
        }
    }).start();
}
```
<h4>Noncompliant code example</h4>

```java
void enqueue2(){
    Thread.startVirtualThread(() -> { // Noncompliant; use a platform thread instead of a virtual one
        if(someCondition){
            synchronizedMethod();
        }else{
            defaultLogic();
        }
    });
}
synchronized void synchronizedMethod(){}
void defaultLogic(){}
```
<h4>Compliant solution</h4>

```java
void enqueue2(){
    new Thread(() -> {
        if(someCondition){
            synchronizedMethod();
        }else{
            defaultLogic();
        }
    }).start();
}
synchronized void synchronizedMethod(){}
void defaultLogic(){}
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> Java Documentation - <a href="https://openjdk.org/jeps/444#:~:text=There%20are%20two,by%20capturing%20carriers">Virtual threads, pinning
  scenarios</a> </li>
</ul>