/*
 * Decompiled with CFR 0.152.
 */
package org.conqat.engine.persistence.config;

import java.io.File;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.conqat.engine.core.configuration.ConfigurationException;
import org.conqat.engine.persistence.config.DatabaseTypeChecker;
import org.conqat.engine.persistence.store.IStorageSystemProvider;
import org.conqat.engine.persistence.store.StorageException;
import org.conqat.engine.persistence.store.leveldb.LevelDBStorageSystemProvider;
import org.conqat.engine.persistence.store.mem.InMemoryStorageSystemProvider;
import org.conqat.engine.persistence.store.rocksdb.RocksDBStorageSystemProvider;
import org.conqat.engine.persistence.store.xodus.XodusStorageSystemProvider;
import org.conqat.lib.commons.assertion.CCSMAssert;
import org.conqat.lib.commons.collections.CollectionUtils;
import org.conqat.lib.commons.collections.UnmodifiableMap;
import org.conqat.lib.commons.enums.EnumUtils;
import org.conqat.lib.commons.string.StringUtils;
import org.conqat.lib.commons.system.SystemUtils;

public class DatabaseConfiguration {
    private static final Logger LOGGER = LogManager.getLogger();
    private static final String DEFAULT_DATABASE = EDatabaseType.MEMORY.name();
    private static final String DATABASE_KEY = "database.type";
    private static final String SHARDING_KEY = "database.sharding";
    private static final String STORAGE_DIRECTORY_KEY = "database.directory";
    private static final String DEFAULT_STORAGE_DIRECTORY = "storage";
    private static final String CACHE_SIZE_KEY = "database.cache-size";
    private static final String DEFAULT_CACHE_SIZE = "512";
    private static final String DETAILS_PREFIX = "database.details.";
    private final EDatabaseType databaseType;
    private final @Nullable String shardingDescription;
    private final File storageDirectory;
    private final int cacheSizeMB;
    private final UnmodifiableMap<String, String> detailSettings;

    public DatabaseConfiguration(EDatabaseType databaseType, @Nullable String shardingDescription, File storageDirectory, int cacheSizeMB, Map<String, String> detailSettings) {
        CCSMAssert.isNotNull((Object)((Object)databaseType), () -> String.format("Expected \"%s\" to be not null", "databaseType"));
        CCSMAssert.isNotNull((Object)storageDirectory, () -> String.format("Expected \"%s\" to be not null", "storageDirectory"));
        CCSMAssert.isNotNull(detailSettings, () -> String.format("Expected \"%s\" to be not null", "detailSettings"));
        this.databaseType = databaseType;
        this.shardingDescription = shardingDescription;
        this.storageDirectory = storageDirectory;
        this.cacheSizeMB = cacheSizeMB;
        this.detailSettings = CollectionUtils.asUnmodifiable(new HashMap<String, String>(detailSettings));
    }

    public static DatabaseConfiguration readFromProperties(Properties properties) throws ConfigurationException {
        String databaseTypeName = properties.getProperty(DATABASE_KEY, DEFAULT_DATABASE);
        EDatabaseType databaseType = (EDatabaseType)EnumUtils.valueOfIgnoreCase(EDatabaseType.class, (String)databaseTypeName);
        if (databaseType == null) {
            throw new IllegalArgumentException("Invalid database type: " + databaseTypeName + ". Valid database types are " + StringUtils.concat((Object[])EDatabaseType.values(), (String)", ").toLowerCase());
        }
        String shardingDescription = properties.getProperty(SHARDING_KEY, "");
        File storageDirectory = new File(properties.getProperty(STORAGE_DIRECTORY_KEY, DEFAULT_STORAGE_DIRECTORY)).getAbsoluteFile();
        DatabaseTypeChecker.checkDatabaseType(storageDirectory, databaseType.name());
        int cacheSizeMB = DatabaseConfiguration.extractIntegerParameter(properties, CACHE_SIZE_KEY, DEFAULT_CACHE_SIZE);
        Map<String, String> detailSettings = properties.stringPropertyNames().stream().filter(key -> key.startsWith(DETAILS_PREFIX)).collect(Collectors.toMap(key -> StringUtils.stripPrefix((String)key, (String)DETAILS_PREFIX), properties::getProperty));
        return new DatabaseConfiguration(databaseType, shardingDescription, storageDirectory, cacheSizeMB, detailSettings);
    }

    public void logConfigSettings() {
        EDatabaseType dbType = this.getDatabaseType();
        LOGGER.info("Using storage: '" + dbType.getDescription() + "'");
        if (dbType.isLocalDiskBased()) {
            LOGGER.info("Using storage directory: '" + this.getStorageDirectory().getAbsolutePath() + "'");
        }
        LOGGER.info("Using cache size of " + this.getCacheSizeMB() + "MB");
    }

    public static int extractIntegerParameter(Properties properties, String propertyKey, String defaultValue) throws ConfigurationException {
        String stringValue = properties.getProperty(propertyKey, defaultValue);
        String errorMessage = "Invalid value for '" + propertyKey + "': " + stringValue;
        try {
            int value = Integer.parseInt(stringValue);
            if (value < 0) {
                throw new ConfigurationException(errorMessage);
            }
            return value;
        }
        catch (NumberFormatException e) {
            throw new ConfigurationException(errorMessage, (Throwable)e);
        }
    }

    public int getCacheSizeMB() {
        return this.cacheSizeMB;
    }

    public EDatabaseType getDatabaseType() {
        return this.databaseType;
    }

    public Optional<String> getShardingDescription() {
        return Optional.ofNullable(this.shardingDescription).filter(Predicate.not(StringUtils::isEmpty));
    }

    public File getStorageDirectory() {
        return this.storageDirectory;
    }

    public DatabaseConfiguration withDatabaseType(EDatabaseType databaseType) {
        return new DatabaseConfiguration(databaseType, this.shardingDescription, this.storageDirectory, this.cacheSizeMB, (Map<String, String>)this.detailSettings);
    }

    public DatabaseConfiguration withStorageDirectory(File storageDirectory) {
        return new DatabaseConfiguration(this.databaseType, this.shardingDescription, storageDirectory, this.cacheSizeMB, (Map<String, String>)this.detailSettings);
    }

    public DatabaseConfiguration withTemporaryStore(File storageDirectory) {
        int updatedCacheSize = Math.max(this.cacheSizeMB / 5, 100);
        return new DatabaseConfiguration(this.databaseType, null, storageDirectory, updatedCacheSize, (Map<String, String>)this.detailSettings);
    }

    public IStorageSystemProvider createStorageSystemProvider(int workerCount) throws StorageException {
        return this.databaseType.createStorageSystemProvider(this, workerCount);
    }

    public static enum EDatabaseType {
        MEMORY("In-memory implementation without persistence.", false){

            @Override
            protected IStorageSystemProvider createStorageSystemProvider(DatabaseConfiguration configuration, int workerCount) {
                return new InMemoryStorageSystemProvider();
            }
        }
        ,
        LEVELDB("LevelDB using native libraries.", true){

            @Override
            protected IStorageSystemProvider createStorageSystemProvider(DatabaseConfiguration configuration, int workerCount) throws StorageException {
                return new LevelDBStorageSystemProvider(configuration.getStorageDirectory(), configuration.getCacheSizeMB());
            }
        }
        ,
        ROCKSDB("RocksDB using native libraries.", true){

            @Override
            protected IStorageSystemProvider createStorageSystemProvider(DatabaseConfiguration configuration, int workerCount) throws StorageException {
                try {
                    return new RocksDBStorageSystemProvider(configuration.getStorageDirectory(), configuration.getCacheSizeMB(), (Map<String, String>)configuration.detailSettings);
                }
                catch (UnsatisfiedLinkError e) {
                    String errorMessage = SystemUtils.getUnsatisfiedLinkageErrorMessage((String)"RocksDB");
                    LOGGER.error(errorMessage, (Throwable)e);
                    System.err.println(errorMessage);
                    throw new RuntimeException(errorMessage, e);
                }
            }
        }
        ,
        XODUS("Xodus in-process pure java database", true){

            @Override
            protected IStorageSystemProvider createStorageSystemProvider(DatabaseConfiguration configuration, int workerCount) throws StorageException {
                return new XodusStorageSystemProvider(configuration.getStorageDirectory(), configuration.getCacheSizeMB(), Integer.parseInt((String)configuration.detailSettings.getOrDefault((Object)"log-file-size-mb", (Object)Integer.toString(512))));
            }
        };

        private final String description;
        private final boolean localDiskBased;

        private EDatabaseType(String description, boolean localDiskBased) {
            this.description = description;
            this.localDiskBased = localDiskBased;
        }

        public String getDescription() {
            return this.description;
        }

        public boolean isLocalDiskBased() {
            return this.localDiskBased;
        }

        protected abstract IStorageSystemProvider createStorageSystemProvider(DatabaseConfiguration var1, int var2) throws StorageException;

        public static EDatabaseType of(String databaseTypeName) {
            EDatabaseType databaseType = (EDatabaseType)EnumUtils.valueOfIgnoreCase(EDatabaseType.class, (String)databaseTypeName);
            if (databaseType == null) {
                throw new IllegalArgumentException("Invalid database type: " + databaseTypeName + ". Valid database types are " + StringUtils.concat((Object[])EDatabaseType.values(), (String)", ").toLowerCase());
            }
            return databaseType;
        }
    }
}

