/*
 * Decompiled with CFR 0.152.
 */
package com.teamscale.core.rest.client;

import java.io.IOException;
import java.time.Duration;
import java.time.LocalDateTime;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.Optional;
import okhttp3.Interceptor;
import okhttp3.Response;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.conqat.lib.commons.date.DurationUtils;

public class HttpRequestRateLimiter
implements Interceptor {
    private static final String RETRY_AFTER_HEADER_NAME = "Retry-After";
    private static final long SLEEP_TIME_MILLIS = DurationUtils.ONE_SECOND.toMillis();
    private static final long TIMEOUT_MINUTES = 10L;
    private static final Duration MAX_RETRY_DELAY = Duration.ofMinutes(5L);
    private static final int STATUS_CODE_TOO_MANY_REQUESTS = 429;
    private static final int STATUS_CODE_SERVICE_UNAVAILABLE = 503;
    private final Duration initialRetryDelay;
    private Duration lastRetryDelay;
    private LocalDateTime nextCallAllowed = LocalDateTime.MIN;
    private final Object lock = new Object();

    public HttpRequestRateLimiter() {
        this(Duration.ofSeconds(5L));
    }

    protected HttpRequestRateLimiter(Duration initialRetryDelay) {
        this.initialRetryDelay = initialRetryDelay;
        this.lastRetryDelay = initialRetryDelay;
    }

    public @NonNull Response intercept(// Could not load outer class - annotation placement on inner may be incorrect
    @NonNull Interceptor.Chain chain) throws IOException {
        try {
            return this.executeCallWithExponentialBackoff(chain);
        }
        catch (InterruptedException e) {
            throw new IOException("Interrupted during rate-limited request " + String.valueOf(chain.request().url()) + ".", e);
        }
    }

    private Response executeCallWithExponentialBackoff(Interceptor.Chain chain) throws IOException, InterruptedException {
        LocalDateTime timeout = LocalDateTime.now().plusMinutes(10L);
        do {
            this.waitUntilNextCallIsAllowed();
            Response httpResponse = chain.proceed(chain.request());
            Optional retryAfter = httpResponse.headers().values(RETRY_AFTER_HEADER_NAME).stream().findFirst();
            if (!this.isRequestRateLimited(httpResponse)) {
                if (retryAfter.isPresent()) {
                    this.updateRetryParameters(this.getRetryDelay(httpResponse, this.lastRetryDelay));
                    return httpResponse;
                }
                this.resetRetryDelay();
                return httpResponse;
            }
            this.updateRetryParameters(this.getRetryDelay(httpResponse, this.lastRetryDelay));
            if (httpResponse.body() == null) continue;
            httpResponse.close();
        } while (LocalDateTime.now().isBefore(timeout));
        throw new IOException("Timeout after waiting for access to rate limited resource " + chain.request().url().host() + " for 10 minutes.");
    }

    protected boolean isRequestRateLimited(Response httpResponse) {
        int statusCode = httpResponse.code();
        return statusCode == 429 || statusCode == 503;
    }

    protected Optional<Duration> getRetryDelayFromHeaders(Response httpResponse) {
        Optional<String> retryAfter = HttpRequestRateLimiter.getRetryAfterHeaderValue(httpResponse);
        return retryAfter.map(this::parseRetryDelayHeaderValue);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void resetRetryDelay() {
        Object object = this.lock;
        synchronized (object) {
            this.lastRetryDelay = this.initialRetryDelay;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void updateRetryParameters(Duration retryDelay) {
        Object object = this.lock;
        synchronized (object) {
            LocalDateTime localThreadNextCallAllowed = LocalDateTime.now().plus(retryDelay);
            if (localThreadNextCallAllowed.isAfter(this.nextCallAllowed)) {
                this.lastRetryDelay = retryDelay;
                this.nextCallAllowed = localThreadNextCallAllowed;
            }
        }
    }

    private void waitUntilNextCallIsAllowed() throws InterruptedException {
        LocalDateTime now = LocalDateTime.now();
        while (now.isBefore(this.nextCallAllowed)) {
            Thread.sleep(SLEEP_TIME_MILLIS);
            now = LocalDateTime.now();
        }
    }

    private Duration getRetryDelay(Response httpResponse, Duration lastRetryDelay) {
        Optional<Duration> duration = this.getRetryDelayFromHeaders(httpResponse);
        if (duration.isPresent()) {
            return duration.get();
        }
        long nextRetryDelay = lastRetryDelay.toSeconds() + HttpRequestRateLimiter.withRandomJitter(lastRetryDelay.toSeconds());
        return DurationUtils.getSmallerDuration((Duration)Duration.ofSeconds(nextRetryDelay), (Duration)MAX_RETRY_DELAY);
    }

    private static long withRandomJitter(long retryDelayMillis) {
        double randomJitter = 0.7 + Math.random() * 0.6;
        return (long)((double)retryDelayMillis * randomJitter);
    }

    protected Duration parseRetryDelayHeaderValue(String retryAfter) {
        try {
            long delayInSeconds = Long.parseLong(retryAfter);
            if (delayInSeconds > 0L) {
                return DurationUtils.getSmallerDuration((Duration)Duration.ofSeconds(delayInSeconds), (Duration)MAX_RETRY_DELAY);
            }
        }
        catch (NumberFormatException e) {
            try {
                ZonedDateTime httpDate = ZonedDateTime.parse(retryAfter, DateTimeFormatter.RFC_1123_DATE_TIME);
                ZonedDateTime now = ZonedDateTime.now();
                if (httpDate.isAfter(now)) {
                    return DurationUtils.getSmallerDuration((Duration)Duration.between(now, httpDate), (Duration)MAX_RETRY_DELAY);
                }
            }
            catch (DateTimeParseException ex) {
                return this.initialRetryDelay;
            }
        }
        return this.initialRetryDelay;
    }

    protected static Optional<String> getRetryAfterHeaderValue(Response httpResponse) {
        return HttpRequestRateLimiter.getHeaderValue(httpResponse, RETRY_AFTER_HEADER_NAME);
    }

    protected static Optional<String> getHeaderValue(Response httpResponse, String headerName) {
        return httpResponse.headers().values(headerName).stream().findFirst();
    }
}

