<h2>Why is this an issue?</h2>
<p>Both the <code>List.Exists</code> method and <code>IEnumerable.Any</code> method can be used to find the first element that satisfies a predicate
in a collection. However, <code>List.Exists</code> can be faster than <code>IEnumerable.Any</code> for <code>List</code> objects, as well as requires
significantly less memory. For small collections, the performance difference may be negligible, but for large collections, it can be noticeable. The
same applies to <code>ImmutableList</code> and arrays too.</p>
<p><strong>Applies to</strong></p>
<ul>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.collections.generic.list-1.exists">List</a> </li>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.array.exists">Array</a> </li>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.collections.immutable.immutablelist-1.exists">ImmutableList</a> </li>
</ul>
<h3>What is the potential impact?</h3>
<p>We measured at least 3x improvement in execution time. For more details see the <code>Benchmarks</code> section from the <code>More info</code>
tab.</p>
<p>Also, no memory allocations were needed for the <code>Exists</code> method, since the search is done in-place.</p>
<h3>Exceptions</h3>
<p>Since <code><a href="https://learn.microsoft.com/en-us/dotnet/framework/data/adonet/ef/language-reference/linq-to-entities">LINQ to
Entities</a></code> relies a lot on <code>System.Linq</code> for <a
href="https://learn.microsoft.com/en-us/dotnet/framework/data/adonet/ef/language-reference/linq-to-entities#query-conversion">query conversion</a>,
this rule won’t raise when used within LINQ to Entities syntaxes.</p>
<h2>How to fix it</h2>
<p>The <code>Exists</code> method is defined on the collection class, and it has the same signature as <code>Any</code> extension method if a
predicate is used. The method can be replaced in place.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>
<pre data-diff-id="1" data-diff-type="noncompliant">
bool ContainsEven(List&lt;int&gt; data) =&gt;
    data.Any(x =&gt; x % 2 == 0);
</pre>
<pre data-diff-id="2" data-diff-type="noncompliant">
bool ContainsEven(int[] data) =&gt;
    data.Any(x =&gt; x % 2 == 0);
</pre>
<h4>Compliant solution</h4>
<pre data-diff-id="1" data-diff-type="compliant">
bool ContainsEven(List&lt;int&gt; data) =&gt;
    data.Exists(x =&gt; x % 2 == 0);
</pre>
<pre data-diff-id="2" data-diff-type="compliant">
bool ContainsEven(int[] data) =&gt;
    Array.Exists(data, x =&gt; x % 2 == 0);
</pre>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.collections.generic.list-1.exists">List&lt;T&gt;.Exists(Predicate&lt;T&gt;)</a>
  </li>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.array.exists">Array.Exists&lt;T&gt;(T[], Predicate&lt;T&gt;)</a> </li>
  <li> <a
  href="https://learn.microsoft.com/en-us/dotnet/api/system.collections.immutable.immutablelist-1.exists">ImmutableList&lt;T&gt;.Exists(Predicate&lt;T&gt;)</a> </li>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/api/system.linq.enumerable.any">Enumerable.Any(Predicate&lt;T&gt;)</a> </li>
  <li> <a href="https://learn.microsoft.com/en-us/dotnet/framework/data/adonet/ef/language-reference/linq-to-entities">LINQ to Entities</a> </li>
</ul>
<h3>Benchmarks</h3>
<table>
  <colgroup>
    <col style="width: 20%;">
    <col style="width: 20%;">
    <col style="width: 20%;">
    <col style="width: 20%;">
    <col style="width: 20%;">
  </colgroup>
  <thead>
    <tr>
      <th>Method</th>
      <th>Runtime</th>
      <th>Mean</th>
      <th>Standard Deviation</th>
      <th>Allocated</th>
    </tr>
  </thead>
  <tbody>
    <tr>
      <td><p>Any</p></td>
      <td><p>.NET 7.0</p></td>
      <td><p>6.670 ms</p></td>
      <td><p>0.1413 ms</p></td>
      <td><p>40004 B</p></td>
    </tr>
    <tr>
      <td><p>Exists</p></td>
      <td><p>.NET 7.0</p></td>
      <td><p>1.364 ms</p></td>
      <td><p>0.0270 ms</p></td>
      <td><p>1 B</p></td>
    </tr>
    <tr>
      <td><p>Any</p></td>
      <td><p>.NET Framework 4.6.2</p></td>
      <td><p>5.380 ms</p></td>
      <td><p>0.0327 ms</p></td>
      <td><p>40128 B</p></td>
    </tr>
    <tr>
      <td><p>Exists</p></td>
      <td><p>.NET Framework 4.6.2</p></td>
      <td><p>1.575 ms</p></td>
      <td><p>0.0348 ms</p></td>
      <td><p>-</p></td>
    </tr>
  </tbody>
</table>
<h4>Glossary</h4>
<ul>
  <li> <a href="https://en.wikipedia.org/wiki/Arithmetic_mean">Mean</a> </li>
  <li> <a href="https://en.wikipedia.org/wiki/Standard_deviation">Standard Deviation</a> </li>
  <li> <a href="https://en.wikipedia.org/wiki/Memory_management">Allocated</a> </li>
</ul>
<p>The results were generated by running the following snippet with <a href="https://github.com/dotnet/BenchmarkDotNet">BenchmarkDotNet</a>:</p>
<pre>
private List&lt;int&gt; data;
private readonly Random random = new Random();

[Params(1_000)]
public int N { get; set; }

[GlobalSetup]
public void Setup() =&gt;
    data = Enumerable.Range(0, N).Select(x =&gt; 43).ToList();

[Benchmark(Baseline = true)]
public void Any()
{
    for (var i = 0; i &lt; N; i++)
    {
        _ = data.Any(x =&gt; x % 2 == 0);          // Enumerable.Any
    }
}

[Benchmark]
public void Exists()
{
    for (var i = 0; i &lt; N; i++)
    {
        _ = data.Exists(x =&gt; x % 2 == 0);       // List&lt;T&gt;.Exists
    }
}
</pre>
<p>Hardware configuration:</p>
<pre>
BenchmarkDotNet=v0.13.5, OS=Windows 10 (10.0.19045.2846/22H2/2022Update)
11th Gen Intel Core i7-11850H 2.50GHz, 1 CPU, 16 logical and 8 physical cores
.NET SDK=7.0.203
  [Host]               : .NET 7.0.5 (7.0.523.17405), X64 RyuJIT AVX2
  .NET 7.0             : .NET 7.0.5 (7.0.523.17405), X64 RyuJIT AVX2
  .NET Framework 4.6.2 : .NET Framework 4.8.1 (4.8.9139.0), X64 RyuJIT VectorSize=256
</pre>

