<p>In an <code>async</code> method, any blocking operations should be avoided.</p>
<h2>Why is this an issue?</h2>
<p>Using a synchronous method instead of its <a href="https://learn.microsoft.com/en-us/dotnet/csharp/asynchronous-programming/">asynchronous</a>
counterpart in an <code>async</code> method blocks the execution and is considered bad practice for several reasons:</p>
<dl>
  <dt>
    Resource Utilization
  </dt>
  <dd>
    <p>Each thread consumes system resources, such as memory. When a thread is blocked, it’s not doing any useful work, but it’s still consuming these
    resources. This can lead to inefficient use of system resources.</p>
  </dd>
  <dt>
    Scalability
  </dt>
  <dd>
    <p>Blocking threads can limit the scalability of your application. In a high-load scenario where many operations are happening concurrently, each
    blocked thread represents a missed opportunity to do useful work. This can prevent your application from effectively handling increased load.</p>
  </dd>
  <dt>
    Performance
  </dt>
  <dd>
    <p>Blocking threads can degrade the performance of your application. If all threads in the thread pool become blocked, new tasks can’t start
    executing until an existing task completes and frees up a thread. This can lead to delays and poor responsiveness.</p>
  </dd>
</dl>
<p>Instead of blocking, it’s recommended to use the <a
href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/await"><code>async</code> operator</a> with async methods. This
allows the system to release the current thread back to the thread pool until the awaited task is complete, improving scalability and
responsiveness.</p>
<h2>How to fix it</h2>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>
<pre data-diff-id="1" data-diff-type="noncompliant">
public async Task Examples(Stream stream, DbSet&lt;Person&gt; dbSet)
{
    stream.Read(array, 0, 1024);            // Noncompliant
    File.ReadAllLines("path");              // Noncompliant
    dbSet.ToList();                         // Noncompliant in Entity Framework Core queries
    dbSet.FirstOrDefault(x =&gt; x.Age &gt;= 18); // Noncompliant in Entity Framework Core queries
}
</pre>
<h4>Compliant solution</h4>
<pre data-diff-id="1" data-diff-type="compliant">
public async Task Examples(Stream stream, DbSet&lt;Person&gt; dbSet)
{
    await stream.ReadAsync(array, 0, 1024);
    await File.ReadAllLinesAsync("path");
    await dbSet.ToListAsync();
    await dbSet.FirstOrDefaultAsync(x =&gt; x.Age &gt;= 18);
}
</pre>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> Microsoft Learn - <a href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/keywords/async">async (C# Reference)</a> </li>
  <li> Microsoft Learn - <a href="https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/await">await operator - asynchronously
  await for a task to complete</a> </li>
</ul>
<h3>Articles &amp; blog posts</h3>
<ul>
  <li> Microsoft Learn - <a href="https://learn.microsoft.com/en-us/dotnet/csharp/asynchronous-programming/#dont-block-await-instead">Asynchronous
  programming with async and await - Don’t block, await instead</a> </li>
  <li> Microsoft Learn - <a
  href="https://learn.microsoft.com/en-us/archive/msdn-magazine/2013/march/async-await-best-practices-in-asynchronous-programming">Async/Await - Best
  Practices in Asynchronous Programming</a> </li>
  <li> Microsoft Developer Blog - <a href="https://devblogs.microsoft.com/pfxteam/asyncawait-faq/">Async/Await FAQ</a> </li>
</ul>

