<h2>Why is this an issue?</h2>
<p>When testing exception via <code>org.junit.rules.ExpectedException</code> any code after the raised exception will not be executed, so adding
subsequent assertions is wrong and misleading. This rule raises an issue when an assertion is done after the "expect(…​)" invocation, only the code
throwing the expected exception should be after "expect(…​)".</p>
<p>You should consider using <a
href="https://github.com/junit-team/junit4/wiki/Exception-testing#using-assertthrows-method">org.junit.Assert.assertThrows</a> instead, it’s available
since JUnit 4.13 and it allows additional subsequent assertions.</p>
<p>Alternatively, you could use <a href="https://github.com/junit-team/junit4/wiki/Exception-testing#trycatch-idiom">try-catch idiom</a> for JUnit
version &lt; 4.13 or if your project does not support lambdas.</p>
<h3>Noncompliant code example</h3>

```java
@Rule
public ExpectedException thrown = ExpectedException.none();

@Test
public void test() throws IndexOutOfBoundsException {
  thrown.expect(IndexOutOfBoundsException.class); // Noncompliant
  Object o = get();
  // This test pass since execution will never get past this line.
  Assert.assertEquals(0, 1);
}

private Object get() {
  throw new IndexOutOfBoundsException();
}
```
<h3>Compliant solution</h3>
<ul>
  <li> For JUnit &gt;= 4.13, use <a
  href="https://github.com/junit-team/junit4/wiki/Exception-testing#using-assertthrows-method">org.junit.Assert.assertThrows</a>: </li>
</ul>

```java
Assert.assertThrows(IndexOutOfBoundsException.class, () -> get());
// This test correctly fails.
Assert.assertEquals(0, 1);
```
<ul>
  <li> For JUnit &lt; 4.13, use the <a href="https://github.com/junit-team/junit4/wiki/Exception-testing#trycatch-idiom">try-catch idiom</a>: </li>
</ul>

```java
try {
  get();
  Assert.fail("Expected an IndexOutOfBoundsException to be thrown");
} catch (IndexOutOfBoundsException e) {}
Assert.assertEquals(0, 1); // Correctly fails.
```
<h2>Resources</h2>
<ul>
  <li> <a href="https://github.com/junit-team/junit4/wiki/Exception-testing">JUnit exception testing documentation</a> </li>
</ul>