<h2>Why is this an issue?</h2>
<p>Regular expressions are used for pattern matching within strings. They can be defined upon special characters, meaning symbols or metacharacters
with a reserved meaning that convey specific instructions to the regex engine. These characters are not treated as literals but have special functions
in defining patterns, among which stand out anchors and disjunctions.</p>
<ul>
  <li> An anchor allows you to match positions in the input string rather than matching specific characters. Anchors help you identify specific
  locations within the string where a pattern should start (<code>^</code>) or end (<code>$</code>). </li>
  <li> A disjunction, also known as alternatives, represented by the vertical bar (<code>|</code>) allows you to specify multiple alternative patterns
  that the regex engine will attempt to match in the input string. </li>
</ul>
<p>Mixing anchors with alternatives in regular expressions can lead to confusion due to their precedence rules. Alternatives (<code>|</code>) have a
lower precedence than anchors (<code>^</code> and <code>$</code>). As a result, if you don’t use non-capturing groups <code>(?:...)</code> to group
the alternatives properly, the anchors might apply to the ends only rather than the entire disjunction, which could not be the initial intent.</p>

```ts
const regex = /^a|b|c$/; // Noncompliant: '^' applies to 'a' and '$' applies to 'c'
```
<p>You should group the disjunction with parentheses denoting non-capturing groups so that the anchors apply to all alternatives.</p>

```ts
const regex = /^(?:a|b|c)$/;
```
<p>Alternatively, you can distribute the anchors to each alternative of the disjunction.</p>

```ts
const regex = /^a$|^b$|^c$/;
```
<p>If the precedence of the operators is understood and the intention is to apply the anchors to only the ends, use parentheses to make it
explicit.</p>

```ts
const regex = /(?:^a)|b|(?:c$)/;
```
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Guide/Regular_expressions">Regular expressions</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Guide/Regular_expressions/Assertions">Assertions</a> </li>
  <li> MDN web docs - <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Regular_expressions/Disjunction">Disjunction:
  <code>|</code></a> </li>
  <li> MDN web docs - <a
  href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Regular_expressions/Non-capturing_group">Non-capturing group:
  <code>(?:...)</code></a> </li>
</ul>