<p>An attacker may trick a user into using a predetermined session identifier. Consequently, this attacker can gain unauthorized access and
impersonate the user’s session. This kind of attack is called session fixation, and protections against it should not be disabled.</p>
<h2>Why is this an issue?</h2>
<p>Session fixation attacks take advantage of the way web applications manage session identifiers. Here’s how a session fixation attack typically
works:</p>
<ul>
  <li> When a user visits a website or logs in, a session is created for them. </li>
  <li> This session is assigned a unique session identifier, stored in a cookie, in local storage, or through URL parameters. </li>
  <li> In a session fixation attack, an attacker tricks a user into using a predetermined session identifier controlled by the attacker. For example,
  the attacker sends the victim an email containing a link with this predetermined session identifier. </li>
  <li> When the victim clicks on the link, the web application does not create a new session identifier but uses this identifier known to the
  attacker. </li>
  <li> At this point, the attacker can hijack and impersonate the victim’s session. </li>
</ul>
<h3>What is the potential impact?</h3>
<p>Session fixation attacks pose a significant security risk to web applications and their users. By exploiting this vulnerability, attackers can gain
unauthorized access to user sessions, potentially leading to various malicious activities. Some of the most relevant scenarios are the following:</p>
<h4>Impersonation</h4>
<p>Once an attacker successfully fixes a session identifier, they can impersonate the victim and gain access to their account without providing valid
credentials. This can result in unauthorized actions, such as modifying personal information, making unauthorized transactions, or even performing
malicious activities on behalf of the victim. An attacker can also manipulate the victim into performing actions they wouldn’t normally do, such as
revealing sensitive information or conducting financial transactions on the attacker’s behalf.</p>
<h4>Data Breach</h4>
<p>If an attacker gains access to a user’s session, they may also gain access to sensitive data associated with that session. This can include
personal information, financial details, or any other confidential data that the user has access to within the application. The compromised data can
be used for identity theft, financial fraud, or other malicious purposes.</p>
<h4>Privilege Escalation</h4>
<p>In some cases, session fixation attacks can be used to escalate privileges within a web application. By fixing a session identifier with higher
privileges, an attacker can bypass access controls and gain administrative or privileged access to the application. This can lead to unauthorized
modifications, data manipulation, or even complete compromise of the application and its underlying systems.</p>
<h2>How to fix it in Passport</h2>
<h3>Code examples</h3>
<p>Upon user authentication, it is crucial to regenerate the session identifier to prevent fixation attacks. Passport provides a mechanism to achieve
this by using the <code>req.session.regenerate()</code> method. By calling this method after successful authentication, you can ensure that each user
is assigned a new and unique session ID.</p>
<h4>Noncompliant code example</h4>

```ts
app.post('/login',
  passport.authenticate('local', { failureRedirect: '/login' }),
  function(req, res) {
    // Noncompliant - no session.regenerate after login
    res.redirect('/');
  });
```
<h4>Compliant solution</h4>

```ts
app.post('/login',
  passport.authenticate('local', { failureRedirect: '/login' }),
  function(req, res) {
    let prevSession = req.session;
    req.session.regenerate((err) => {
      Object.assign(req.session, prevSession);
      res.redirect('/');
    });
  });
```
<h3>How does this work?</h3>
<p>The protection works by ensuring that the session identifier, which is used to identify and track a user’s session, is changed or regenerated
during the authentication process.</p>
<p>Here’s how session fixation protection typically works:</p>
<ol>
  <li> When a user visits a website or logs in, a session is created for them. This session is assigned a unique session identifier, which is stored
  in a cookie or passed through URL parameters. </li>
  <li> In a session fixation attack, an attacker tricks a user into using a predetermined session identifier controlled by the attacker. This allows
  the attacker to potentially gain unauthorized access to the user’s session. </li>
  <li> To protect against session fixation attacks, session fixation protection mechanisms come into play during the authentication process. When a
  user successfully authenticates, this mechanism generates a new session identifier for the user’s session. </li>
  <li> The old session identifier, which may have been manipulated by the attacker, is invalidated and no longer associated with the user’s session.
  This ensures that any attempts by the attacker to use the fixed session identifier are rendered ineffective. </li>
  <li> The user is then assigned the new session identifier, which is used for subsequent requests and session tracking. This new session identifier
  is typically stored in a new session cookie or passed through URL parameters. </li>
</ol>
<p>By regenerating the session identifier upon authentication, session fixation protection helps ensure that the user’s session is tied to a new,
secure identifier that the attacker cannot predict or control. This mitigates the risk of an attacker gaining unauthorized access to the user’s
session and helps maintain the integrity and security of the application’s session management process.</p>
<h2>Resources</h2>
<h3>Documentation</h3>
<ul>
  <li> Express.js Documentation - <a href="http://expressjs.com/en/resources/middleware/session.html">express-session</a> </li>
</ul>
<h3>Articles &amp; blog posts</h3>
<ul>
  <li> <a href="https://medium.com/passportjs/fixing-session-fixation-b2b68619c51d">Fixing Session Fixation</a> </li>
</ul>
<h3>Standards</h3>
<ul>
  <li> OWASP - <a href="https://owasp.org/Top10/A07_2021-Identification_and_Authentication_Failures/">Top 10 2021 Category A7 - Identification and
  Authentication Failures</a> </li>
  <li> OWASP - <a href="https://owasp.org/www-project-top-ten/2017/A2_2017-Broken_Authentication">Top 10 2017 Category A2 - Broken Authentication</a>
  </li>
  <li> <a href="https://owasp.org/www-community/attacks/Session_fixation">OWASP Sesssion Fixation</a> </li>
  <li> CWE - <a href="https://cwe.mitre.org/data/definitions/384">CWE-384 - Session Fixation</a> </li>
  <li> STIG Viewer - <a href="https://stigviewer.com/stig/application_security_and_development/2023-06-08/finding/V-222579">Application Security and
  Development: V-222579</a> - Applications must use system-generated session identifiers that protect against session fixation. </li>
  <li> STIG Viewer - <a href="https://stigviewer.com/stig/application_security_and_development/2023-06-08/finding/V-222582">Application Security and
  Development: V-222582</a> - The application must not re-use or recycle session IDs. </li>
</ul>