Calling `Thread.sleep` in test methods can lead to unreliable test outcomes.
Consider utilizing test-specific methods to manage timing dependencies.

# What Does This Check Look For?

This check identifies the presence of `Thread.sleep` within the body of JUnit test methods.

# Why is This a Problem?

Introducing `Thread.sleep` in tests can result in several issues:
- It makes test execution time longer unnecessarily.
- The reliability of test results might be compromised due to dependencies on timing, leading to flaky tests.
- It doesn't properly simulate the conditions under which the code is supposed to run, thereby lowering the quality of tests.

Relying on `Thread.sleep` presupposes knowledge about the exact time a certain operation takes, which is unpredictable and varies across different environments and conditions.

# How Can I Resolve This?

Replace `Thread.sleep` with more robust and reliable synchronization mechanisms. For instance:
- Use test-specific methods like `WebDriverWait` for Selenium to wait for the expected condition.
- Use libraries like Awaitility that can wait for specific conditions without relying on fixed sleep times.
- Employ callback mechanisms or use `Future`, `CompletionStage`, or other concurrency primitives from the Java Concurrency API.

# Examples

## Selenium example

### Non-Compliant

```java
@Test
public void testWithSelenium() throws InterruptedException {
	WebElement element = driver.findElement(By.id("elementId"));
	Thread.sleep(1000);
	element.click();
}
```

### Compliant

```java
@Test
public void testWithSelenium() {
	WebDriverWait wait = new WebDriverWait(driver, 10);
	WebElement element = wait.until(ExpectedConditions.elementToBeClickable(By.id("elementId")));
	element.click();
}
```

## Generic example

### Non-Compliant

```java
@Test
public void updatesCustomerStatus() throws InterruptedException {
	messageBroker.publishMessage(updateCustomerStatusMessage);
	Thread.sleep(1000);
	assertEquals(customerStatus(), "Waiting for customer response");
}
```

### Compliant

```java
@Test
public void updatesCustomerStatus() {
	messageBroker.publishMessage(updateCustomerStatusMessage);
	await().atMost(5, SECONDS).until(customerStatusIsUpdated());
	assertEquals(customerStatus(), "Waiting for customer response");
}
```

# Where Can I Learn More?
- [Selenium Waiting Strategies](https://www.selenium.dev/documentation/webdriver/waits/)
- [Playwright Auto-waiting](https://playwright.dev/docs/actionability)
- [Awaitility Usage wiki page](https://github.com/awaitility/awaitility/wiki/Usage)