Using improper annotation for nested classes in JUnit5 may cause them not to be executed during build tests.
Properly annotate inner classes with `@Nested` and avoid using `@Nested` for static nested classes to ensure correct behavior during test execution.

# What Does This Check Look For?

This check reports a finding when an inner class with test methods is not annotated with `@Nested`, or when a static nested class is unnecessarily annotated with `@Nested`.

# Why is This a Problem?

If an inner class containing tests is not annotated with `@Nested`, JUnit5 will not execute its tests during the build process, only in the IDE.
Conversely, applying `@Nested` to a static nested class is unnecessary and can lead to confusion, as JUnit5 will not manage it along with its enclosing class.

# How Can I Resolve This?

1. Annotate inner classes containing test methods with `@Nested`.
2. Avoid annotating static nested classes with `@Nested`.

Ensuring that the `@Nested` annotation is applied correctly will allow JUnit5 to execute all intended test methods during the build.

# Examples
## Non-Compliant

```java
import org.junit.jupiter.api.Test;

class MyJunit5Test {
  @Test
  void test() { /* ... */ }

  class InnerClassTest { // Noncompliant, missing @Nested annotation
    @Test
    void test() { /* ... */ }
  }

  @Nested
  static class StaticNestedClassTest { // Noncompliant, invalid usage of @Nested annotation
    @Test
    void test() { /* ... */ }
  }
}
```

## Compliant

```java
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.Nested;

class MyJunit5Test {
  @Test
  void test() { /* ... */ }

  @Nested
  class InnerClassTest {
    @Test
    void test() { /* ... */ }
  }

  static class StaticNestedClassTest {
    @Test
    void test() { /* ... */ }
  }
}
```

# Where Can I Learn More?
- [JUnit 5 User Guide: Nested Tests](https://junit.org/junit5/docs/current/user-guide/#writing-tests-nested)

