The check descriptions for Security Code Scan (SCS) were extracted from the **Rules** section of the SCS website: https://security-code-scan.github.io/#Rules

The following steps were used to retrieve the descriptions (as of 2023-11):

1. Save the website as local `html` file and delete everything inside the `<body>` except the **Rules** section.
2. Use the script below to extract the individual rules and convert them to markdown
3. Copy the `.md` files from the output folder to Teamscale's repository (this folder)

### Rule Extraction Script

This small TypeScript project was run with Bun (a JS/TS runner): https://bun.sh 

The three files below have to be copied into a folder, followed by

1. Check the **TODO** regarding the input path in `main.ts`
2. `bun install`
3. `bun main.ts`
4. Check the contents of the `output` directory

**package.json** 
``` json
{
  "name": "scs-description-parser",
  "module": "main.ts",
  "type": "module",
  "devDependencies": {
    "bun-types": "latest"
  },
  "peerDependencies": {
    "typescript": "^5.0.0",
    "linkedom": "0.16.4",
    "node-html-markdown": "1.3.0"
  }
}
```
**tsconfig.json**
``` json
{
  "name": "scs-description-parser",
  "module": "main.ts",
  "type": "module",
  "devDependencies": {
    "bun-types": "latest"
  },
  "peerDependencies": {
    "typescript": "^5.0.0",
    "linkedom": "0.16.4",
    "node-html-markdown": "1.3.0"
  }
}
```
**main.ts**
``` typescript
import { HTMLElement, HTMLHeadingElement, parseHTML } from "linkedom";
import { NodeHtmlMarkdown } from "node-html-markdown";

// TODO: Check this path to your local copy of the SCS website
const scsWebsiteLocal = "../scs.html";

const html = await Bun.file(scsWebsiteLocal).text();

const { document } = parseHTML(html);

const markdownByRuleId: Record<string, string> = {};

document.querySelectorAll("h3").forEach((h3: HTMLHeadingElement) => {
  let ruleAsMarkdown = "";
  const ruleId = h3.innerHTML.match(/(SCS\d+)/)![1];
  if (!ruleId) {
    throw new Error("Could not extract rule id SCS.... from " + h3.innerHTML);
  }
  let nextElement: HTMLElement | null = h3;
  while ((nextElement = nextElement.nextSibling)) {
    if (nextElement.tagName === "H3") {
      break;
    }
    if (nextElement?.outerHTML) {
      ruleAsMarkdown += "\n" + nextElement.outerHTML;
    }
  }
  markdownByRuleId[ruleId] =
    "###" +
    NodeHtmlMarkdown.translate(h3.innerHTML) +
    "\n\n" +
    NodeHtmlMarkdown.translate(ruleAsMarkdown) +
    "\n\n" +
    "Source: [Security Code Scan](https://security-code-scan.github.io/#Rules)";
});

Object.keys(markdownByRuleId).forEach(async (ruleId) => {
  await Bun.write("output/" + ruleId + ".md", markdownByRuleId[ruleId]);
  console.log(markdownByRuleId[ruleId][0]);
});
``` 
