<h2>Why is this an issue?</h2>
<p>Classes annotated as <code>@Controller</code> in Spring are responsible for handling incoming web requests. When annotating methods or the entire
controller with <code>@ResponseBody</code>, the return value of said methods will be serialized and set as the response body. In other words, it tells
the Spring framework that this method does not produce a view. This mechanism is commonly used to create API endpoints.</p>
<p>Spring provides <code>@RestController</code> as a convenient annotation to replace the combination of <code>@Controller</code> and
<code>@ResponseBody</code>. The two are functionally identical, so the single annotation approach is preferred.</p>
<p>This rule will raise an issue on a class that is annotated with <code>@Controller</code> if:</p>
<ul>
  <li> the class is also annotated with <code>@ResponseBody</code> or </li>
  <li> all methods in said class are annotated with <code>@ResponseBody</code>. </li>
</ul>
<h2>How to fix it</h2>
<p>Replace the <code>@Controller</code> annotation with the <code>@RestController</code> annotation and remove all <code>@ResponseBody</code>
annotations from the class and its methods.</p>
<h3>Code examples</h3>
<h4>Noncompliant code example</h4>

```java
@Controller
@ResponseBody
public class MyController {
    @GetMapping("/hello")
    public String hello() {
        return "Hello World!";
    }
}
```
<h4>Compliant solution</h4>

```java
@RestController
public class MyController {
    @GetMapping("/hello")
    public String hello() {
        return "Hello World!";
    }
}
```
<h4>Noncompliant code example</h4>

```java
@Controller
public class MyController {
    @ResponseBody
    @GetMapping("/hello")
    public String hello() {
        return "Hello World!";
    }

    @ResponseBody
    @GetMapping("/foo")
    public String foo() {
        return "Foo";
    }
}
```
<h4>Compliant solution</h4>

```java
@RestController
public class MyController {
    @GetMapping("/hello")
    public String hello() {
        return "Hello World!";
    }

    @GetMapping("/foo")
    public String foo() {
        return "Foo";
    }
}
```
<h2>Resources</h2>
<h3>Articles &amp; blog posts</h3>
<ul>
  <li> Spring Guides - <a href="https://spring.io/guides/gs/rest-service/">Building a RESTful Web Service</a> </li>
  <li> Baeldung - <a href="https://www.baeldung.com/spring-controller-vs-restcontroller">The Spring @Controller and @RestController Annotations</a>
  </li>
  <li> Baeldung - <a href="https://www.baeldung.com/spring-request-response-body">Spring’s RequestBody and ResponseBody Annotations</a> </li>
</ul>